from datetime import datetime
from pydantic import BaseModel, Field
from typing import Optional

# ====================================================
# Base Model for Maintenance Record
# ====================================================
class MaintenanceRecordBase(BaseModel):
    vehicle_id: int = Field(..., description="Unique identifier for the vehicle undergoing maintenance")
    maintenance_type: str = Field(..., description="Type of maintenance (e.g., routine, emergency, repair)")
    scheduled_date: datetime = Field(..., description="Scheduled date for the maintenance")
    log_details: str = Field(..., description="Detailed description or log of the maintenance performed")
    maintenance_status: Optional[str] = Field("pending", description="Current status of maintenance (e.g., pending, completed, cancelled)")
    created_date: datetime = Field(default_factory=datetime.utcnow, description="Timestamp when the maintenance record was created")
    
    class Config:
        orm_mode = True

# ====================================================
# Model for Creating a New Maintenance Record
# ====================================================
class MaintenanceCreate(MaintenanceRecordBase):
    pass

# ====================================================
# Model for Updating an Existing Maintenance Record
# ====================================================
class MaintenanceUpdate(BaseModel):
    vehicle_id: Optional[int] = Field(None, description="Updated vehicle identifier")
    maintenance_type: Optional[str] = Field(None, description="Updated type of maintenance")
    scheduled_date: Optional[datetime] = Field(None, description="Updated scheduled date for maintenance")
    log_details: Optional[str] = Field(None, description="Updated maintenance log details")
    maintenance_status: Optional[str] = Field(None, description="Updated status of maintenance")

# ====================================================
# Model for Reading a Maintenance Record
# ====================================================
class MaintenanceRead(MaintenanceRecordBase):
    maintenance_record_id: int = Field(..., description="Unique identifier for the maintenance record")

# ====================================================
# Model for Searching Maintenance Records
# ====================================================
class MaintenanceSearch(BaseModel):
    vehicle_id: Optional[int] = Field(None, description="Filter by vehicle identifier")
    maintenance_type: Optional[str] = Field(None, description="Filter by maintenance type")
    scheduled_date_start: Optional[datetime] = Field(None, description="Filter by scheduled date (start of range)")
    scheduled_date_end: Optional[datetime] = Field(None, description="Filter by scheduled date (end of range)")
