from enum import Enum
from datetime import datetime
from typing import Optional, Dict, Any
from pydantic import BaseModel, Field

# ====================================================
# Enum for Entity Types Generating the Log
# ====================================================
class LogEntityType(str, Enum):
    driver = "driver"
    vehicle = "vehicle"
    job = "job"
    route = "route"
    service = "service"
    maintenance = "maintenance"
    # Extend with additional entity types as needed

# ====================================================
# Base Model for a Log Entry
# ====================================================
class LogBase(BaseModel):
    event_type: str = Field(..., description="Label for the event (e.g., 'speeding', 'job_started')")
    timestamp: datetime = Field(default_factory=datetime.utcnow, description="Time when the event occurred")
    entity_type: Optional[LogEntityType] = Field(None, description="Type of the entity generating the event")
    entity_id: Optional[int] = Field(None, description="Identifier of the entity generating the event")
    description: Optional[str] = Field(None, description="A brief description of the event")
    metadata: Dict[str, Any] = Field(default_factory=dict, description="Additional event-specific data (e.g., sensor readings, cost factors)")

    class Config:
        orm_mode = True

# ====================================================
# Model for Creating a Log Entry
# ====================================================
class LogCreate(LogBase):
    pass

# ====================================================
# Model for Reading a Log Entry
# ====================================================
class LogRead(LogBase):
    log_id: int = Field(..., description="Unique identifier for the log entry")
