# fleet.py
from pydantic import BaseModel, Field, EmailStr
from datetime import datetime, date
from typing import Optional
from ..common import Address

class Vehicle(BaseModel):
    vehicle_id: int
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    registration_number: str = Field(..., description="Vehicle registration number")
    vin_number: str = Field(..., description="Vehicle Identification Number")
    vehicle_model_id: str = Field(..., description="Reference to the vehicle model record")
    color: str = Field(..., description="Vehicle color")
    year: int = Field(..., description="Manufacturing year")
    vehicle_image: str = Field(..., description="Image URL or file path for the vehicle")
    insurance_number: Optional[str] = Field(None, description="Insurance number")
    insurance_expiry: Optional[date] = Field(None, description="Insurance expiry date")
    vendor_id: str = Field(..., description="Identifier for the vendor")
    dealer_id: str = Field(..., description="Identifier for the dealer")
    status: bool = Field(..., description="Active status (consider enum for more statuses)")
    created_date: datetime = Field(default_factory=datetime.utcnow, description="Record creation timestamp")
    updated_date: datetime = Field(default_factory=datetime.utcnow, description="Last updated timestamp")

class VehicleModel(BaseModel):
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    manufacturer_name: str = Field(..., description="Name of the manufacturer")
    vehicle_type: str = Field(..., description="Type of vehicle (e.g., normal, autonomous, drone)")
    fuel_type: str = Field(..., description="Fuel type (e.g., petrol, diesel, electric)")
    is_autonomous: bool = Field(..., description="True if the vehicle is autonomous")
    can_fly: bool = Field(..., description="True if the vehicle can fly (e.g., drone capabilities)")
    capacity_jobs: int = Field(..., description="Maximum number of concurrent jobs")
    number_of_wheels: int = Field(..., description="Number of wheels")
    max_speed: Optional[int] = Field(None, description="Maximum speed")
    engine_capacity: Optional[int] = Field(None, description="Engine capacity (e.g., in CC)")


class Vendor(BaseModel):
    vendor_id: int
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    vendor_name: str = Field(..., description="Name of the vendor/dealer")
    address: Address = Field(..., description="Structured address information")
    phone: str = Field(..., description="Phone number")
    contact_person_name: str = Field(..., description="Name of the contact person")
    email: EmailStr = Field(..., description="Email address")
    vendor_type: str = Field(..., description='Vendor type (e.g., "vendor" or "dealer")')
    logo: str = Field(..., description="URL for the vendor logo")
    license_number: str = Field(..., description="License number")
