import React, { useState, useEffect } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { Helmet } from 'react-helmet';

// ----------------- VideoBackground Component ----------------- //
const VideoBackground = ({ sources, height }) => {
  const [currentIndex, setCurrentIndex] = useState(0);
  const [nextIndex, setNextIndex] = useState(
    sources.length > 1 ? 1 : 0
  );
  const [fade, setFade] = useState(false);

  useEffect(() => {
    // Every 10 seconds, trigger a fade transition
    const interval = setInterval(() => {
      setFade(true);
      // After fade transition, update the indices
      setTimeout(() => {
        setCurrentIndex(nextIndex);
        setNextIndex((prev) => (prev + 1) % sources.length);
        setFade(false);
      }, 1000); // 1s fade duration
    }, 10000);
    return () => clearInterval(interval);
  }, [nextIndex, sources.length]);

  return (
    <div
      style={{
        position: 'absolute',
        top: 0,
        left: 0,
        width: '100%',
        height: height,
        overflow: 'hidden',
      }}
    >
      {/* Current Video */}
      <video
        autoPlay
        muted
        loop
        playsInline
        style={{
          position: 'absolute',
          top: '50%',
          left: '50%',
          width: '100%',
          height: height,
          objectFit: 'cover',
          transform: 'translate(-50%, -50%)',
        }}
      >
        <source src={sources[currentIndex]} type="video/mp4" />
        Your browser does not support the video tag.
      </video>
      {/* Next Video with crossfade */}
      <video
        autoPlay
        muted
        loop
        playsInline
        style={{
          position: 'absolute',
          top: '50%',
          left: '50%',
          width: '100%',
          height: height,
          objectFit: 'cover',
          transform: 'translate(-50%, -50%)',
          opacity: fade ? 1 : 0,
          transition: 'opacity 1s ease-in-out',
        }}
      >
        <source src={sources[nextIndex]} type="video/mp4" />
        Your browser does not support the video tag.
      </video>
    </div>
  );
};

// ----------------- Dynamic Heading Component ----------------- //
const DynamicHeading = () => {
  const texts = [
    "Transport",
    "Logistics",
    "Workforce",
    "Operations",
    "Courier",
  ];
  
  const colors = [
    "#FF5733",
    "#3357FF",
    "#FF33A1",
    "#F1C40F",
    "#8E44AD",
    "#33FF57",
  ];

  const [index, setIndex] = useState(0);

  useEffect(() => {
    const interval = setInterval(() => {
      setIndex((prevIndex) => (prevIndex + 1) % texts.length);
    }, 3000);
    return () => clearInterval(interval);
  }, [texts.length]);

  return (
    <h1 style={{ margin: '0', color:'white', fontWeight:'300', }}>
      Transform Your{" "}
      <span style={{ color: colors[index], transition: "color 0.5s ease" }}>
        {texts[index]}
      </span>{" "} Ops
    </h1>
  );
};

const movexPlatforms = [
  {
    id: 'dispatch',
    title: 'Dispatch',
    description:
      'Leverage AI-powered dispatching to automatically and manually assign jobs in real time, ensuring optimal resource allocation and responsiveness.',
    icon: 'local_shipping',
  },
  {
    id: 'fleet-tracking',
    title: 'Fleet Tracking',
    description:
      'Manage and monitor your fleet with precision. Track vehicles, schedule maintenance, and optimize routes using intelligent insights.',
    icon: 'navigation',
  },
  {
    id: 'workforce',
    title: 'Workforce',
    description:
      'Efficiently manage diverse teams and allocate jobs dynamically with AI, empowering your workforce to perform at its best.',
    icon: 'group',
  },
  {
    id: 'booking-engine',
    title: 'Booking Engine',
    description:
      'Deploy customizable booking engines on web and mobile that attract new customers and streamline appointments seamlessly.',
    icon: 'event_available',
  },
  {
    id: 'mobile-apps',
    title: 'Mobile Apps',
    description:
      'Enhance mobility with dedicated apps for customers, drivers, and workforce—keeping everyone connected and informed on the go.',
    icon: 'smartphone',
  },
  {
    id: 'ai-analytics',
    title: 'AI & Advanced Analytics',
    description:
      'Integrate state-of-the-art LLMs and AI models to power every aspect of your operations—from booking and dispatch to routing and customer interaction.',
    icon: 'memory',
  },
];

const realtimeApplications = [
  {
    id: 'real-time-ops',
    title: 'Real-Time Operations',
    description:
      'Experience the impact of immediate, AI-driven decisions across your business—optimizing processes and enhancing customer satisfaction.',
    icon: 'speed',
  },
  {
    id: 'dynamic-routing',
    title: 'Dynamic Routing',
    description:
      'Utilize intelligent routing solutions that adapt in real time, reducing delays and improving overall efficiency.',
    icon: 'alt_route',
  },
  {
    id: 'customer-interaction',
    title: 'Enhanced Customer Interaction',
    description:
      'Engage every customer with personalized experiences powered by AI, turning interactions into meaningful engagements.',
    icon: 'thumb_up',
  },
];

const interactiveSections = [
  {
    show: true,
    title: 'Log in to Manage Your Operations',
    text:
      'Access your personalized dashboard to review past operations, manage bookings, and obtain tailored insights for optimizing your logistics.',
    button: { text: 'Log in', link: '/login' },
    extraText: 'Don’t have an account?',
    extraLink: { text: 'Sign up', link: '/signup' },
    image:
      'https://www.quickanddirtytips.com/wp-content/uploads/2022/05/ezgif.com-gif-maker-3.jpg',
  },
  {
    title: 'Book a Move in Minutes',
    text:
      'Experience a seamless booking process designed for modern logistics. Schedule moves effortlessly and let MOVEX handle dispatch and coordination with advanced AI.',
    button: { text: 'Book a Move', link: '/booking/move' },
    image:
      "https://static.blog.bolt.eu/LIVE/wp-content/uploads/2023/02/13170947/wait-and-save-1-1-1024x536.jpeg",
  },
  {
    title: 'Operate & Earn on Your Schedule',
    text:
      'Join our network of operators and drivers to maximize efficiency and profitability. Enjoy flexible scheduling and advanced tools that adapt to your needs.',
    button: { text: 'Start Operating', link: '/operate' },
    image:
      "https://cdlschool.com/hubfs/Imported_Blog_Media/iStock-1358424193.jpeg",
  },
  {
    title: 'Enterprise Transport Solutions',
    text:
      'Transform your company’s logistics with a comprehensive platform designed for scalable operations. Manage resources, track performance, and optimize every move with precision.',
    button: { text: 'Explore Solutions', link: '/enterprise' },
    image:
      "https://www.uber-assets.com/image/upload/f_auto,q_auto:eco,c_fill,h_768,w_1152/v1730197725/assets/0f/48c7ba-da13-4fdc-b54c-42878042f513/original/Airport-Fall.png",
  },
];

// Array of video sources for the banner background
const videoSources = [ "/slow3.mp4" ]; // Add additional sources as needed

const WebHome = () => {
  const navigate = useNavigate();

  function handleClick() {
    navigate('/signup');
  }

  return (
    <>
      <Helmet>
        <title>MOVEX: Shaping the Future of Moving People, Goods & Things</title>
        <meta
          name="description"
          content="MOVEX is an AI-powered platform revolutionizing the way you move people, goods, and things. Leverage advanced dispatch, fleet tracking, workforce management, and more to transform your operations."
        />
      </Helmet>

      {/* Banner / Hero Section with Background Videos */}
      <div className="banner-container" style={{ position: 'relative', overflow: 'hidden', width: '100%', height: '500px', marginBottom: '50px', top: 0 }}>
        <VideoBackground sources={videoSources} height="500px" />
        {/* Semi-transparent overlay is now included inside VideoBackground if needed */}
        <section className="main-section" style={{ position: 'relative', zIndex: '1', color: '#fff', padding: '250px 20px', textAlign: 'center' }}>
          <DynamicHeading />
          <h3 style={{ color:'white' }}>
            Harness the power of advanced AI to revolutionize the movement of people, goods, and things.
          </h3>
          <br />
          <div>
            <Link className="buttonstyle" to="/signup">
              Get Started Now
            </Link>
            &nbsp;&nbsp;
            <Link to="/experience" className="homebutton">
              <span className="material-symbols-outlined" style={{ padding: '0px', fontSize: '20px' }}>
                insights
              </span>
              Explore Platforms
            </Link>
          </div>
        </section>
      </div>

      {/* Rest of the Page Content */}
      <div className="website-container" style={{ position: 'relative', overflow: 'hidden', padding: '40px 0' }}>
        <center>
          <div className="content-container">
            {/* Platform / Item Section */}
            <div>
              <h1>How MOVEX Works</h1>
            </div>
            <p>
              Discover our suite of AI-driven platforms designed to streamline every aspect of your operations—from dispatch to dynamic routing.
              Each solution works in harmony to propel your business into the future.
            </p>
            <section className="applications-section">
              {movexPlatforms.map((platform) => (
                <div className="application" key={platform.id}>
                  <span className="material-symbols-outlined application-icon">
                    {platform.icon}
                  </span>
                  <h2>{platform.title}</h2>
                  <p>{platform.description}</p>
                </div>
              ))}
            </section>
            <div className="section-gap">
              <button onClick={handleClick}>Learn How It Works</button>
            </div>

            {/* Interactive Sections */}
            {interactiveSections.map((section, index) => (
              <section className="interactive-section" key={index}>
                <div
                  className="interactive-content"
                  style={{
                    display: 'flex',
                    alignItems: 'center',
                    flexWrap: 'wrap',
                    margin: '40px 0',
                    flexDirection: index % 2 === 0 ? 'row' : 'row-reverse',
                  }}
                >
                  {/* Image side */}
                  <div
                    className="interactive-image"
                    style={{ flex: '1', minWidth: '300px', padding: '20px' }}
                  >
                    <img
                      src={section.image}
                      alt={section.title}
                      style={{ width: '100%', borderRadius: '8px' }}
                    />
                  </div>
                  {/* Text side */}
                  <div
                    className="interactive-text"
                    style={{ flex: '1', minWidth: '300px', padding: '20px', textAlign: 'left' }}
                  >
                    <h1>{section.title}</h1>
                    <p style={{ whiteSpace: 'pre-line' }}>{section.text}</p>
                    <Link className="buttonstyle" to={section.button.link}>
                      {section.button.text}
                    </Link>
                    {section.extraText && section.extraLink && (
                      <p style={{ marginTop: '10px' }}>
                        {section.extraText}&nbsp;
                        <Link to={section.extraLink.link}>{section.extraLink.text}</Link>
                      </p>
                    )}
                  </div>
                </div>
              </section>
            ))}

            {/* Real-Time Applications Section */}
            <div>
              <h1>Real-Time Applications</h1>
            </div>
            <p>
              Experience immediate, intelligent decision-making.
              Our real-time applications empower your business to adapt and respond with agility.
            </p>
            <section className="applications-section">
              {realtimeApplications.map((app) => (
                <div className="application" key={app.id}>
                  <span className="material-symbols-outlined application-icon">
                    {app.icon}
                  </span>
                  <h2>{app.title}</h2>
                  <p>{app.description}</p>
                </div>
              ))}
            </section>
            <div className="section-gap">
              <button onClick={handleClick}>Discover More</button>
            </div>
          </div>
        </center>
      </div>
    </>
  );
};

export default WebHome;
