import React, { useState, useEffect } from 'react';
import { BrowserRouter as Router, Routes, Route, useLocation } from 'react-router-dom';
import CommonRoutes from './CommonRoutes';
import SaaSRoutes from './SaaSRoutes';
import PlatformRoutes from './PlatformRoutes';
import WebsiteRoutes from './WebsiteRoutes';
import {useSelector, useDispatch } from 'react-redux';
import { setRBAC } from '../../state/RBACSlice';

const websitePaths = [
  "/", "/about", "/contact", "/how-it-works", "/applications", 
  "/pricing", "/help", "/privacy", "/terms", "/interact", 
  "/website-generativeai-chatbot", "/generativeai-team-assist", "/hello"
];

const AppRoutes = () => {
  return (
    <Router>
      <Routes>
        <Route path="/*" element={<RouteSelector />} />
      </Routes>
    </Router>
  );
};

const RouteSelector = () => {
  const location = useLocation();
  const [isWebsiteRoute, setIsWebsiteRoute] = useState(false);
  const dispatch = useDispatch();
  const rbacLoaded = useSelector(state => state.rbac.loaded);

  useEffect(() => {
    try {
      const roleRights = JSON.parse(localStorage.getItem('role_rights') || '{}');
      const roleFeatures = JSON.parse(localStorage.getItem('role_features') || '[]');
      dispatch(setRBAC({ roleRights, roleFeatures }));
    } catch (e) {
      console.error('Failed to load RBAC from localStorage', e);
    }
  }, [dispatch]);

  useEffect(() => {
    setIsWebsiteRoute(
      websitePaths.includes(location.pathname) ||
      websitePaths.some(path => location.pathname.startsWith(`${path}/`))
    );
  }, [location.pathname]);

  // ✅ Wait until RBAC is loaded
  if (!rbacLoaded) {
    return <div className="loading-screen">Loading...</div>; // or null, spinner, whatever
  }

  const currentUserRole = localStorage.getItem('roles') || 100;

  if (isWebsiteRoute) {
    return <WebsiteRoutes />;
  } else if (currentUserRole <= 99) {
    return (
      <>
        <CommonRoutes />
        <SaaSRoutes />
      </>
    );
  } else {
    return (
      <>
        <CommonRoutes />
        <PlatformRoutes />
      </>
    );
  }
};
  

export default AppRoutes;
