import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import CommonTaskList from '../../common/CommonTaskList';
import TaskBar from '../../common/TaskListBar';

const TAB_CONFIG = [
  { id: 'details', label: 'Details', icon: 'info' },
//   { id: 'workforce', label: 'Workforce', icon: 'people' },
//   { id: 'tasks', label: 'Tasks', icon: 'assignment' },
//   { id: 'schedule', label: 'Schedule', icon: 'schedule' }
];

const WorkforceCategoryView = ({ workforceCategoryId, onClose }) => {
  const [categoryData, setCategoryData] = useState(null);
  const [workforce, setWorkforce] = useState([]);
  const [tasks, setTasks] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('details');
  const [hoveredTab, setHoveredTab] = useState(null);

  useEffect(() => {
    const fetchData = async () => {
      try {
        const categoryResponse = await axios.get(`/workforcecategory/${workforceCategoryId}`);
        setCategoryData(categoryResponse.data);

        if (categoryResponse.data) {
          try {
            const [workforceResponse, tasksResponse] = await Promise.all([
              axios.get(`/workforce?category_id=${workforceCategoryId}`),
              axios.get(`/tasks?workforce_category=${workforceCategoryId}`)
            ]);
            setWorkforce(workforceResponse.data || []);
            setTasks(tasksResponse.data || []);
          } catch (err) {
            console.error("Error fetching additional data:", err);
          }
        }

        setLoading(false);
      } catch (err) {
        console.error("Error fetching category details:", err);
        setError('Error fetching workforce category details.');
        setLoading(false);
      }
    };

    fetchData();
  }, [workforceCategoryId]);

  const renderTabContent = () => {
    if (!categoryData) return null;

    switch (activeTab) {
      case 'details':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '12px',
                color: '#333'
              }}>
                info
              </span>
              Category Details
            </h4>

            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <p><strong>Category Name:</strong> {categoryData.category_name}</p>
              <p><strong>Description:</strong> {categoryData.description || 'N/A'}</p>
              <p><strong>Shift Hours:</strong> {categoryData.shift_hours}</p>
              <p><strong>Capacity Jobs:</strong> {categoryData.capacity_jobs}</p>
              <p><strong>Created Date:</strong> {new Date(categoryData.created_date).toLocaleDateString()}</p>
              {/* <p><strong>Last Updated:</strong> {new Date(categoryData.updated_date).toLocaleDateString()}</p> */}
            </div>
          </div>
        );

      case 'workforce':
        return (
          <div style={{ marginTop: '15px' }}>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                people
              </span>
              Workforce Members
            </h4>
            {workforce.length > 0 ? (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <table style={{ width: '100%', borderCollapse: 'collapse' }}>
                  <thead>
                    <tr style={{ borderBottom: '1px solid #eee' }}>
                      <th style={{ textAlign: 'left', padding: '8px' }}>Name</th>
                      <th style={{ textAlign: 'left', padding: '8px' }}>Email</th>
                      <th style={{ textAlign: 'left', padding: '8px' }}>Status</th>
                      <th style={{ textAlign: 'left', padding: '8px' }}>Last Active</th>
                    </tr>
                  </thead>
                  <tbody>
                    {workforce.map((member, index) => (
                      <tr key={index} style={{ borderBottom: index < workforce.length - 1 ? '1px solid #eee' : 'none' }}>
                        <td style={{ padding: '8px' }}>{member.name}</td>
                        <td style={{ padding: '8px' }}>{member.email}</td>
                        <td style={{ padding: '8px' }}>
                          <span style={{
                            display: 'inline-block',
                            padding: '2px 8px',
                            borderRadius: '12px',
                            backgroundColor: member.status === 'active' ? '#e6f7ee' : '#f5f5f5',
                            color: member.status === 'active' ? '#00a854' : '#666'
                          }}>
                            {member.status}
                          </span>
                        </td>
                        <td style={{ padding: '8px' }}>
                          {member.last_active ? new Date(member.last_active).toLocaleString() : 'Never'}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            ) : (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '20px',
                marginTop: '10px',
                textAlign: 'center',
                color: '#666'
              }}>
                No workforce members found in this category
              </div>
            )}
          </div>
        );

      case 'tasks':
        return (
          <div style={{ marginTop: '15px' }}>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                assignment
              </span>
              Category Tasks
            </h4>
            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <CommonTaskList
                presetFilters={{ workforce_category: workforceCategoryId }}
                rowsPerPage={5}
                initialSearchQuery=""
              />
            </div>
          </div>
        );

      case 'schedule':
        return (
          <div style={{ marginTop: '15px' }}>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                schedule
              </span>
              Category Schedule
            </h4>
            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <TaskBar workforceCategoryId={workforceCategoryId} />
              <div style={{ marginTop: '20px' }}>
                <p>Schedule visualization would go here</p>
                <p>Showing shifts and assignments for this workforce category</p>
              </div>
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading workforce category data...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }

  return (
    <Modal onClose={onClose}>
      <div style={{ height: '100%', overflow: 'hidden' }}>
        {/* Header */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          gap: '20px',
          padding: '0px 10px 10px',
          borderBottom: '1px solid #e0e0e0'
        }}>
          <div style={{
            width: '60px',
            height: '60px',
            borderRadius: '50%',
            backgroundColor: '#e0e0e0',
            color: '#000',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            fontSize: '20px',
            fontWeight: 'bold'
          }}>
            {categoryData.category_name?.charAt(0)?.toUpperCase() || 'W'}
          </div>
          <div>
            <h3 style={{ margin: '0px', fontWeight: 'bold', fontSize: '20px', color: 'black' }}>
              {categoryData.category_name}
            </h3>
            <p style={{ marginTop: '5px', fontSize: '16px', color: 'black', fontWeight: 'bold' }}>
              ID : #{categoryData. workforce_category_id}
            </p>
          </div>
        </div>

        {/* Tabs */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '20px 6px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '2px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && <span>{tab.label}</span>}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-20px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Content */}
        <div style={{ padding: '0 20px', overflow: 'hidden' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default WorkforceCategoryView;

