

import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import WorkforcecategoryView from './WorkforcecategoryView';
import WorkforcecategoryUpdate from './WorkforcecategoryUpdate';
import WorkforcecategoryDB from './WorkforcecategoryDB';
import CommonData from '../../common/CommonData';

const WORKFORCE_CATEGORIES_HEADER_TO_KEY_MAP = {
  "Category Name": {
    key: "category_name",
    label: "Category Name",
    sortable: true,
    clickable: true // ✅ Make this field clickable
  },
  "Description": { key: "description", label: "Description", sortable: false },
  "Status": { key: "status", label: "Status", sortable: false },
  "Created Date": { key: "created_date", label: "Created Date", sortable: false },
};

const WorkforceCategoryList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery
  });

  const [selectedWorkforceCategory, setSelectedWorkforceCategory] = useState(null);

  const enums = CommonData();

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' },
        { label: 'Blocked', value: 'blocked' }
      ]
    },
    { name: 'reg_date_from', type: 'date', placeholder: 'From Date' },
    { name: 'reg_date_to', type: 'date', placeholder: 'To Date' },
  ];

  const actionFormats = [
    { label: "view", modal: true, url: "/workforcecategory/view/{id}" },
    { label: "edit", modal: true, url: "/workforcecategory/update/{id}" },
    { label: "delete", modal: true, url: "/workforcecategory/delete/{id}" },
  ];

  const handleActionClick = (workforceCategoryId, action) => {
    if (["view", "edit", "delete"].includes(action.label)) {
      setSelectedWorkforceCategory({ workforceCategoryId, modalType: action.label });
    }
  };

  const handleCellClick = (header, item) => {
    if (header === "Category Name") {
      const categoryId = item.workforcecategory_id || item.id;
      if (categoryId) {
        console.log('✅ Category Name clicked → Opening view modal for ID:', categoryId);
        setSelectedWorkforceCategory({
          workforceCategoryId: categoryId,
          modalType: 'view'
        });
      } else {
        console.warn('⚠️ Missing workforcecategory_id in item:', item);
      }
    }
  };

  const handleSearch = (params) => {
    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      reg_date_from: params.reg_date_from || '',
      reg_date_to: params.reg_date_to || '',
      status: params.status || '',
      skip: 0
    }));
  };

  const handlePageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage
    }));
  };

  const closeModal = () => setSelectedWorkforceCategory(null);

  const buildFetchUrl = () => {
    let url = `/workforcecategory/list/?skip=${searchParams.skip}&limit=${searchParams.limit}`;
    if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
    if (searchParams.reg_date_from) url += `&from=${searchParams.reg_date_from}`;
    if (searchParams.reg_date_to) url += `&to=${searchParams.reg_date_to}`;
    if (searchParams.status) url += `&status=${searchParams.status}`;
    return url;
  };

  return (
    <div>
      <SearchPage
        fetchDataUrl={buildFetchUrl()}
        ListComponent={List}
        itemKeyMapping={WORKFORCE_CATEGORIES_HEADER_TO_KEY_MAP}
        dataKey="workforce_categories"
        identifierKey="workforcecategory_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}  
        searchFields={searchFields}
        onSearch={handleSearch}
        onPageChange={handlePageChange}
      />

      {selectedWorkforceCategory?.modalType === "view" && (
        <WorkforcecategoryView
          workforceCategoryId={selectedWorkforceCategory.workforceCategoryId}
          onClose={closeModal}
        />
      )}

      {selectedWorkforceCategory?.modalType === "edit" && (
        <WorkforcecategoryUpdate
          workforceCategoryId={selectedWorkforceCategory.workforceCategoryId}
          onClose={closeModal}
          onUpdateSuccess={() => console.log("Workforce category updated")}
        />
      )}

      {selectedWorkforceCategory?.modalType === "delete" && (
        <WorkforcecategoryDB
          workforceCategoryId={selectedWorkforceCategory.workforceCategoryId}
          onClose={closeModal}
          onDeleteSuccess={() => console.log("Workforce category deleted")}
        />
      )}
    </div>
  );
};

export default WorkforceCategoryList;
