import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import CommonTaskList from '../../common/CommonTaskList';
import TaskBar from '../../common/TaskListBar';
import '../../themes/blue/Workforceview.css';


const TAB_CONFIG = [
  { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
  { id: 'tasks', label: 'Tasks', icon: 'assignment' },
  { id: 'skills', label: 'Skills', icon: 'build' },
  { id: 'schedules', label: 'Schedules', icon: 'calendar_today' },
  { id: 'vehicles', label: 'Vehicles', icon: 'directions_car' },
  { id: 'timesheet', label: 'Timesheet', icon: 'schedule' },
  { id: 'logs', label: 'Logs', icon: 'list_alt' },
  { id: 'info', label: 'Info', icon: 'info' }
];


const WorkforceView = ({ workforceId, onClose }) => {
  const [workforceData, setWorkforceData] = useState(null);
  const [categoryName, setCategoryName] = useState('');
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');
  const [hoveredTab, setHoveredTab] = useState(null);


  useEffect(() => {
    const fetchData = async () => {
      try {
        const res = await axios.get(`/workforce/${workforceId}`);
        setWorkforceData(res.data);


        if (res.data.workforce_category_id) {
          const categoryRes = await axios.get(`/workforcecategory/${res.data.workforce_category_id}`);
          setCategoryName(categoryRes.data.category_name);
        }


        setLoading(false);
      } catch (error) {
        setError('Error loading details');
        setLoading(false);
      }
    };


    fetchData();
  }, [workforceId]);


  const renderTabContent = () => {
    if (!workforceData) return null;


    switch (activeTab) {
      case 'dashboard':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>dashboard</span>
              Workforce summary
            </h4>
            <p><strong>Name:</strong> {workforceData.first_name} {workforceData.last_name}</p>
            <p><strong>Status:</strong> {workforceData.status}</p>
            <p><strong>Category:</strong> {categoryName}</p>
            <p><strong>Join Date:</strong> {new Date(workforceData.join_date).toLocaleString()}</p>


            <TaskBar workforceId={workforceId} />


            <h4 style={{ marginTop: '1rem' }}>Recent Activity</h4>
            <p><strong>Last 2 Tasks:</strong> (placeholder)</p>
            <p><strong>Last Login:</strong> {workforceData.last_login || '-'}</p>
          </div>
        );


      case 'tasks':
        return (
          <div className="tab-content">
            <h4><span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>assignment</span>Workforce Tasks</h4>
            <div className="task-list-container">
              <CommonTaskList
                presetFilters={{ workforce_id: workforceId }}
                rowsPerPage={5}
                initialSearchQuery=""
              />
            </div>
          </div>
        );


      case 'skills':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>build</span>
              Skills & Certifications
            </h4>
            <div className="skills-grid">
              <div className="skill-card">
                <h5>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>directions_car</span>
                  Driving License
                </h5>
                <p><strong>ID:</strong> {workforceData.license_id || '--'}</p>
                <p><strong>Expiry:</strong> {workforceData.license_expiry || '--'}</p>
              </div>
              <div className="skill-card">
                <h5>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>build</span>
                  Special Skills
                </h5>
                <p>Forklift Operation</p>
                <p>Hazardous Material Handling</p>
              </div>
            </div>
          </div>
        );


      case 'schedules':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>calendar_today</span>
              Work Schedule
            </h4>
            <p><strong>Shift Timing:</strong> {workforceData.shift_timing}</p>
            <div className="calendar-placeholder">
              <p>Calendar view of scheduled shifts</p>
            </div>
          </div>
        );


      case 'vehicles':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>directions_car</span>
              Assigned Vehicles
            </h4>
            {/* <p><strong>Current Vehicle:</strong> {workforceData.assigned_fleet || 'N/A'}</p> */}
            <div className="vehicle-history">
              <h5>Vehicle History</h5>
              <ul>
                <li>Vehicle A (Jan 2023 - Present)</li>
                <li>Vehicle B (Aug 2022 - Dec 2022)</li>
              </ul>
            </div>
          </div>
        );


      case 'timesheet':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>schedule</span>
              Timesheet
            </h4>
            <div className="timesheet-table">
              <table>
                <thead>
                  <tr>
                    <th>Date</th>
                    <th>Check In</th>
                    <th>Check Out</th>
                    <th>Hours</th>
                  </tr>
                </thead>
                <tbody>
                  <tr>
                    <td>2023-05-01</td>
                    <td>08:00 AM</td>
                    <td>05:30 PM</td>
                    <td>9.5</td>
                  </tr>
                  <tr>
                    <td>2023-04-30</td>
                    <td>08:15 AM</td>
                    <td>06:00 PM</td>
                    <td>9.75</td>
                  </tr>
                </tbody>
              </table>
            </div>
          </div>
        );


      case 'logs':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>list_alt</span>
              Active Logs
            </h4>
            <div className="log-entries">
              <div className="log-entry">
                <p><strong>2023-05-01 08:00</strong> - Checked in</p>
              </div>
              <div className="log-entry">
                <p><strong>2023-05-01 12:30</strong> - Started task #1234</p>
              </div>
              <div className="log-entry">
                <p><strong>2023-05-01 17:30</strong> - Checked out</p>
              </div>
            </div>
          </div>
        );


      case 'info':
        return (
          <div className="tab-content">
            <div className="profile-grid">
              <div className="profile-column">
                <h4>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>info</span>
                  Personal Information
                </h4>
                <div className="detail-row"><strong>First Name:</strong> {workforceData.first_name}</div>
                <div className="detail-row"><strong>Last Name:</strong> {workforceData.last_name}</div>
                <div className="detail-row"><strong>Email:</strong> {workforceData.email_id}</div>
                <div className="detail-row"><strong>Mobile:</strong> {workforceData.mobile_number}</div>
                <div className="detail-row"><strong>Gender:</strong> {workforceData.gender}</div>
                <div className="detail-row"><strong>Date of Birth:</strong> {new Date(workforceData.date_of_birth).toLocaleDateString()}</div>
                <div className="detail-row"><strong>Blood Group:</strong> {workforceData.blood_group}</div>
              </div>
              <div className="profile-column">
                <h4>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>info</span>
                  Work Details
                </h4>
                <div className="detail-row"><strong>Join Date:</strong> {new Date(workforceData.join_date).toLocaleString()}</div>
                <div className="detail-row"><strong>Status:</strong> {workforceData.status}</div>
                <div className="detail-row"><strong>Category:</strong> {categoryName}</div>
                <div className="detail-row"><strong>Shift Timing:</strong> {workforceData.shift_timing}</div>
                <div className="detail-row"><strong>Emergency Contact:</strong> {workforceData.emergency_contact}</div>
                <div className="detail-row"><strong>Address:</strong> {workforceData.address}</div>
              </div>
            </div>


            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>info</span>
              Document Details
            </h4>
            <div className="document-grid">
              <div className="detail-row"><strong>License ID:</strong> {workforceData.license_id || '--'}</div>
              {/* <div className="detail-row"><strong>License Expiry:</strong> {workforceData.license_expiry || '--'}</div> */}
              <div className="detail-row"><strong>National ID:</strong> {workforceData.national_id || '--'}</div>
              {/* <div className="detail-row"><strong>PCO Licence:</strong> {workforceData.pco_expiry || '--'}</div> */}
              {/* <div className="detail-row"><strong>Insurance Number:</strong> {workforceData.insurance_number || '--'}</div>
              <div className="detail-row"><strong>National Insurance:</strong> {workforceData.national_insurance || '--'}</div> */}
            </div>
          </div>
        );


      default:
        return null;
    }
  };


  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading workforce data...</div>
      </Modal>
    );
  }


  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }


  return (
    <Modal onClose={onClose}>
      <div className="fixed top-0 right-0 w-full md:w-[55%] lg:w-[45%] xl:w-[40%] h-full bg-white shadow-lg z-50 overflow-hidden">
        {/* Header Section */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          padding: '10px 10px 20px',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px'
        }}>
          <div style={{ display: 'flex', alignItems: 'center' }}>
            <div style={{
              width: '60px',
              height: '60px',
              borderRadius: '50%',
              backgroundColor: '#e0e0e0',
              color: '#000',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              fontSize: '20px',
              fontWeight: 'bold',
              marginRight: '16px'
            }}>
              {workforceData.first_name?.charAt(0).toUpperCase()}
            </div>
            <div>
              <div style={{
                fontWeight: '600',
                fontSize: '20px',
                lineHeight: '1.4',
                color: 'black'
              }}>
                {workforceData.first_name} {workforceData.last_name}
              </div>
              <div style={{
                fontSize: '16px',
                color: 'black',
                marginTop: '10px',
                fontWeight: 'bold'
              }}>
                ID: #{workforceData.w_id}
              </div>
            </div>
          </div>
        </div>


        {/* Tabs Section */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '0px 10px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '20px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px',
                transition: 'min-width 0.2s ease'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && (
                <span>{tab.label}</span>
              )}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-1px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>


        {/* Tab Content */}
        <div style={{ maxHeight: 'calc(100vh - 250px)', overflowY: 'auto', padding: '0 20px' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};


export default WorkforceView;



// import React, { useState, useEffect } from 'react';
// import axios from 'axios';
// import Modal from '../../Layouts/Modal';
// import CommonTaskList from '../../common/CommonTaskList';
// import TaskBar from '../../common/TaskListBar';
// import '../../themes/blue/Workforceview.css';

// const TAB_CONFIG = [
//   { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
//   { id: 'tasks', label: 'Tasks', icon: 'assignment' },
//   { id: 'skills', label: 'Skills', icon: 'build' },
//   { id: 'schedules', label: 'Schedules', icon: 'calendar_today' },
//   { id: 'vehicles', label: 'Vehicles', icon: 'directions_car' },
//   { id: 'timesheet', label: 'Timesheet', icon: 'schedule' },
//   { id: 'logs', label: 'Logs', icon: 'list_alt' },
//   { id: 'info', label: 'Info', icon: 'info' }
// ];

// const WorkforceView = ({ workforceId, workforceData: initialData, onClose }) => {
//   const [workforceData, setWorkforceData] = useState(initialData || null);
//   const [categoryName, setCategoryName] = useState('');
//   const [loading, setLoading] = useState(!initialData);
//   const [error, setError] = useState('');
//   const [activeTab, setActiveTab] = useState('dashboard');
//   const [hoveredTab, setHoveredTab] = useState(null);

//   useEffect(() => {
//     if (!initialData) {
//       const fetchData = async () => {
//         try {
//           const res = await axios.get(`/workforce/${workforceId}`);
//           setWorkforceData(res.data);

//           if (res.data.workforce_category_id) {
//             const categoryRes = await axios.get(`/workforcecategory/${res.data.workforce_category_id}`);
//             setCategoryName(categoryRes.data.category_name);
//           }
//         } catch (error) {
//           setError('Error loading workforce details');
//         } finally {
//           setLoading(false);
//         }
//       };

//       fetchData();
//     } else {
//       if (initialData.workforce_category_id) {
//         axios.get(`/workforcecategory/${initialData.workforce_category_id}`)
//           .then(res => setCategoryName(res.data.category_name))
//           .catch(() => console.error('Error loading category'));
//       }
//       setLoading(false);
//     }
//   }, [workforceId, initialData]);

//   const renderNoDataMessage = (message = "No data available") => (
//     <div className="no-data-message">
//       <span className="material-symbols-outlined">info</span>
//       <p>{message}</p>
//     </div>
//   );

//   const formatDate = (dateString) => {
//     if (!dateString) return '--';
//     try {
//       return new Date(dateString).toLocaleString();
//     } catch {
//       return '--';
//     }
//   };

//   const renderTabContent = () => {
//     if (!workforceData) return null;

//     switch (activeTab) {
//       case 'dashboard':
//         return (
//           <div className="tab-content">
//             <h4>
//               <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>dashboard</span>
//               Workforce Summary
//             </h4>
//             <p><strong>Name:</strong> {workforceData.first_name} {workforceData.last_name}</p>
//             <p><strong>Status:</strong> {workforceData.status || '--'}</p>
//             <p><strong>Category:</strong> {categoryName || '--'}</p>
//             <p><strong>Join Date:</strong> {formatDate(workforceData.join_date)}</p>

//             <TaskBar workforceId={workforceId} />

//             <h4 style={{ marginTop: '1rem' }}>Recent Activity</h4>
//             {workforceData.last_login ? (
//               <p><strong>Last Login:</strong> {formatDate(workforceData.last_login)}</p>
//             ) : renderNoDataMessage("No recent activity found")}
//           </div>
//         );

//       case 'tasks':
//         return (
//           <div className="tab-content">
//             <h4><span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>assignment</span>Workforce Tasks</h4>
//             <div className="task-list-container">
//               <CommonTaskList
//                 presetFilters={{ workforce_id: workforceId }}
//                 rowsPerPage={5}
//                 initialSearchQuery=""
//               />
//             </div>
//           </div>
//         );

//       case 'skills':
//         const hasSkills = workforceData.skills?.length > 0;
//         const hasLicense = workforceData.license_id;
        
//         return (
//           <div className="tab-content">
//             <h4>
//               <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>build</span>
//               Skills & Certifications
//             </h4>
//             <div className="skills-grid">
//               {hasLicense ? (
//                 <div className="skill-card">
//                   <h5>
//                     <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>directions_car</span>
//                     Driving License
//                   </h5>
//                   <p><strong>ID:</strong> {workforceData.license_id}</p>
//                   {workforceData.license_expiry && (
//                     <p><strong>Expiry:</strong> {formatDate(workforceData.license_expiry)}</p>
//                   )}
//                 </div>
//               ) : renderNoDataMessage("No license information available")}

//               {hasSkills ? (
//                 <div className="skill-card">
//                   <h5>
//                     <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>build</span>
//                     Special Skills
//                   </h5>
//                   <ul>
//                     {workforceData.skills.map((skill, index) => (
//                       <li key={index}>{skill}</li>
//                     ))}
//                   </ul>
//                 </div>
//               ) : renderNoDataMessage("No skills information available")}
//             </div>
//           </div>
//         );

//       case 'schedules':
//         return (
//           <div className="tab-content">
//             <h4>
//               <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>calendar_today</span>
//               Work Schedule
//             </h4>
//             {workforceData.shift_timing ? (
//               <>
//                 <p><strong>Shift Timing:</strong> {workforceData.shift_timing}</p>
//                 {renderNoDataMessage("No detailed schedule available")}
//               </>
//             ) : renderNoDataMessage("No schedule information available")}
//           </div>
//         );

//       case 'vehicles':
//         return (
//           <div className="tab-content">
//             <h4>
//               <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>directions_car</span>
//               Assigned Vehicles
//             </h4>
//             {workforceData.assigned_fleet ? (
//               <>
//                 <p><strong>Current Vehicle:</strong> {workforceData.assigned_fleet}</p>
//                 {renderNoDataMessage("No vehicle history available")}
//               </>
//             ) : renderNoDataMessage("No vehicles assigned")}
//           </div>
//         );

//       case 'timesheet':
//         return (
//           <div className="tab-content">
//             <h4>
//               <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>schedule</span>
//               Timesheet
//             </h4>
//             {renderNoDataMessage("No timesheet data available")}
//           </div>
//         );

//       case 'logs':
//         return (
//           <div className="tab-content">
//             <h4>
//               <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>list_alt</span>
//               Activity Logs
//             </h4>
//             {renderNoDataMessage("No log entries available")}
//           </div>
//         );

//       case 'info':
//         return (
//           <div className="tab-content">
//             <div className="profile-grid">
//               <div className="profile-column">
//                 <h4>
//                   <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>info</span>
//                   Personal Information
//                 </h4>
//                 <div className="detail-row"><strong>First Name:</strong> {workforceData.first_name}</div>
//                 <div className="detail-row"><strong>Last Name:</strong> {workforceData.last_name}</div>
//                 <div className="detail-row"><strong>Email:</strong> {workforceData.email_id || '--'}</div>
//                 <div className="detail-row"><strong>Mobile:</strong> {workforceData.mobile_number || '--'}</div>
//                 <div className="detail-row"><strong>Gender:</strong> {workforceData.gender || '--'}</div>
//                 <div className="detail-row"><strong>Date of Birth:</strong> {formatDate(workforceData.date_of_birth)}</div>
//                 <div className="detail-row"><strong>Blood Group:</strong> {workforceData.blood_group || '--'}</div>
//               </div>
//               <div className="profile-column">
//                 <h4>
//                   <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>info</span>
//                   Work Details
//                 </h4>
//                 <div className="detail-row"><strong>Join Date:</strong> {formatDate(workforceData.join_date)}</div>
//                 <div className="detail-row"><strong>Status:</strong> {workforceData.status || '--'}</div>
//                 <div className="detail-row"><strong>Category:</strong> {categoryName || '--'}</div>
//                 <div className="detail-row"><strong>Shift Timing:</strong> {workforceData.shift_timing || '--'}</div>
//                 <div className="detail-row"><strong>Emergency Contact:</strong> {workforceData.emergency_contact || '--'}</div>
//                 <div className="detail-row"><strong>Address:</strong> {workforceData.address || '--'}</div>
//               </div>
//             </div>

//             <h4>
//               <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>info</span>
//               Document Details
//             </h4>
//             <div className="document-grid">
//               <div className="detail-row"><strong>License ID:</strong> {workforceData.license_id || '--'}</div>
//               <div className="detail-row"><strong>License Expiry:</strong> {workforceData.license_expiry ? formatDate(workforceData.license_expiry) : '--'}</div>
//               <div className="detail-row"><strong>National ID:</strong> {workforceData.national_id || '--'}</div>
//               <div className="detail-row"><strong>PCO Licence:</strong> {workforceData.pco_expiry || '--'}</div>
//               <div className="detail-row"><strong>Insurance Number:</strong> {workforceData.insurance_number || '--'}</div>
//               <div className="detail-row"><strong>National Insurance:</strong> {workforceData.national_insurance || '--'}</div>
//             </div>
//           </div>
//         );

//       default:
//         return null;
//     }
//   };

//   if (loading) {
//     return (
//       <Modal onClose={onClose}>
//         <div style={{ padding: '20px', textAlign: 'center' }}>Loading workforce data...</div>
//       </Modal>
//     );
//   }

//   if (error) {
//     return (
//       <Modal onClose={onClose}>
//         <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
//       </Modal>
//     );
//   }

//   return (
//     <Modal onClose={onClose}>
//       <div className="fixed top-0 right-0 w-full md:w-[55%] lg:w-[45%] xl:w-[40%] h-full bg-white shadow-lg z-50 overflow-hidden">
//         {/* Header Section */}
//         <div style={{
//           display: 'flex',
//           alignItems: 'center',
//           padding: '10px 10px 20px',
//           borderBottom: '1px solid #e0e0e0',
//           marginBottom: '20px'
//         }}>
//           <div style={{ display: 'flex', alignItems: 'center' }}>
//             <div style={{
//               width: '60px',
//               height: '60px',
//               borderRadius: '50%',
//               backgroundColor: '#e0e0e0',
//               color: '#000',
//               display: 'flex',
//               alignItems: 'center',
//               justifyContent: 'center',
//               fontSize: '20px',
//               fontWeight: 'bold',
//               marginRight: '16px'
//             }}>
//               {workforceData.first_name?.charAt(0).toUpperCase()}
//             </div>
//             <div>
//               <div style={{
//                 fontWeight: '600',
//                 fontSize: '20px',
//                 lineHeight: '1.4',
//                 color: 'black'
//               }}>
//                 {workforceData.first_name} {workforceData.last_name}
//               </div>
//               <div style={{
//                 fontSize: '16px',
//                 color: 'black',
//                 marginTop: '10px',
//                 fontWeight: 'bold'
//               }}>
//                 ID: #{workforceData.w_id || workforceData.workforce_id}
//               </div>
//             </div>
//           </div>
//         </div>

//         {/* Tabs Section */}
//         <div style={{
//           display: 'flex',
//           borderBottom: '1px solid #e0e0e0',
//           marginBottom: '20px',
//           padding: '0px 10px'
//         }}>
//           {TAB_CONFIG.map(tab => (
//             <button
//               key={tab.id}
//               style={{
//                 padding: '20px 16px',
//                 border: 'none',
//                 backgroundColor: 'transparent',
//                 cursor: 'pointer',
//                 fontSize: '14px',
//                 fontWeight: activeTab === tab.id ? '400' : '400',
//                 color: activeTab === tab.id ? '#34c759' : '#333',
//                 position: 'relative',
//                 whiteSpace: 'nowrap',
//                 minWidth: activeTab === tab.id ? '100px' : '50px',
//                 textAlign: 'center',
//                 display: 'flex',
//                 alignItems: 'center',
//                 justifyContent: 'center',
//                 gap: '8px',
//                 transition: 'min-width 0.2s ease'
//               }}
//               onClick={() => setActiveTab(tab.id)}
//               onMouseEnter={() => setHoveredTab(tab.id)}
//               onMouseLeave={() => setHoveredTab(null)}
//             >
//               <span className="material-symbols-outlined" style={{
//                 color: activeTab === tab.id ? '#34c759' : '#333',
//                 fontSize: '20px'
//               }}>
//                 {tab.icon}
//               </span>
//               {(activeTab === tab.id || hoveredTab === tab.id) && (
//                 <span>{tab.label}</span>
//               )}
//               {activeTab === tab.id && (
//                 <div style={{
//                   position: 'absolute',
//                   bottom: '-1px',
//                   left: '0',
//                   right: '0',
//                   height: '2px',
//                   backgroundColor: '#34c759'
//                 }} />
//               )}
//             </button>
//           ))}
//         </div>

//         {/* Tab Content */}
//         <div style={{ maxHeight: 'calc(100vh - 250px)', overflowY: 'auto', padding: '0 20px' }}>
//           {renderTabContent()}
//         </div>
//       </div>
//     </Modal>
//   );
// };

// export default WorkforceView;