import React, { useState, useEffect } from 'react';
import axios from 'axios';
import WorkforceForm from './WorkforceForm';
import Modal from '../../Layouts/Modal';
import { workforceSchema, workforceUiSchema } from './WorkforceForm';

const dataURLtoFile = (dataurl, filename) => {
    const arr = dataurl.split(',');
    const mime = arr[0].match(/:(.*?);/)[1];
    const bstr = atob(arr[1]);
    let n = bstr.length;
    const u8arr = new Uint8Array(n);
    while (n--) {
        u8arr[n] = bstr.charCodeAt(n);
    }
    return new File([u8arr], filename, { type: mime });
};

const WorkforceUpdate = ({ workforceId, onClose, onUpdateSuccess }) => {
    const [workforceData, setWorkforceData] = useState(null);
    const [categories, setCategories] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState('');

    useEffect(() => {
        const fetchData = async () => {
            try {
                const [workforceRes, categoriesRes] = await Promise.all([
                    axios.get(`/workforce/${workforceId}`),
                    axios.get('/workforcecategory/list/')
                ]);

                const categoriesData = categoriesRes.data?.workforce_categories || [];
                setCategories(categoriesData);

                setWorkforceData({
                    ...workforceRes.data,
                    date_of_birth: workforceRes.data.date_of_birth?.split('T')[0],
                    join_date: workforceRes.data.join_date?.split('T')[0]
                });
            } catch (err) {
                setError('Error loading data');
                console.error('Data fetch error:', err);
            } finally {
                setLoading(false);
            }
        };
        fetchData();
    }, [workforceId]);

    const categoryOptions = Array.isArray(categories) ? categories : [];

    const modifiedSchema = {
        ...workforceSchema,
        properties: {
            ...workforceSchema.properties,
            workforce_category_id: {
                type: "string",
                title: "Workforce Category",
                enum: categoryOptions.map(cat => cat.workforcecategory_id),
                enumNames: categoryOptions.map(cat => cat.category_name),
            }
        }
    };

    const handleSubmit = (formData) => {
        const formDataToSend = new FormData();
        
        Object.keys(formData).forEach(key => {
            if (key !== 'photo') {
                formDataToSend.append(key, formData[key]);
            }
        });

        if (formData.photo && formData.photo.startsWith('data:')) {
            const photoFile = dataURLtoFile(formData.photo, `workforce-${Date.now()}.png`);
            formDataToSend.append('photo', photoFile);
        }

        axios.put(`/workforce/${workforceId}`, formDataToSend, {
            headers: {
                'Content-Type': 'multipart/form-data'
            }
        })
        .then(res => {
            alert("Workforce updated successfully!");
            onUpdateSuccess?.(res.data);
            onClose?.();
        })
        .catch(err => {
            console.error('Update error:', err.response?.data);
            alert("Error updating workforce: " + 
                (err.response?.data?.detail || "Unknown error"));
        });
    };

    return (
        <Modal title="Edit Workforce" onClose={onClose}>
            {loading && <div>Loading...</div>}
            {error && <div className="error">{error}</div>}
            {workforceData && (
                <WorkforceForm 
                    initialData={workforceData}
                    onSubmit={handleSubmit}
                    schema={modifiedSchema}
                    uiSchema={workforceUiSchema}
                    key={categories.length}
                />
            )}
        </Modal>
    );
};

export default WorkforceUpdate;