import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import WorkforceView from './WorkforceView';
import WorkforceUpdate from './WorkforceUpdate';
import WorkforceDB from './WorkforceDB';
import CommonData from '../../common/CommonData';
import FleetAssignmentForm from './FleetAssignmentForm';
import WorkforceCategoryView from '../Workforcecategory/WorkforcecategoryView';
import Skills from '../../adminpages/Skills';

const WorkforceList = ({ initialSearchQuery = '', rowsPerPage = 10, accountId }) => {
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery
  });

  const [selectedWorkforce, setSelectedWorkforce] = useState(null);
  const [selectedCategory, setSelectedCategory] = useState(null);
  const [selectedSkill, setSelectedSkill] = useState(null);

  const enums = CommonData();

  const searchFields = [
    { name: 'join_date_from', type: 'date', placeholder: 'From Date' },
    { name: 'join_date_to', type: 'date', placeholder: 'To Date' },
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' },
        { label: 'Block', value: 'block' }
      ]
    },
  ];

  const WORKFORCE_HEADER_MAP = {
    "First Name": {
      key: "first_name",
      label: "First Name",
      sortable: true,
      clickable: true
    },
    "Last Name": { key: "last_name", label: "Last Name", sortable: true },
    "Workforce Category": { 
      key: "workforce_category_name", 
      label: "Workforce Category", 
      sortable: true,
      clickable: true,
      idKey: "workforce_category_id"
    },
    "Skills": { 
      key: "skills_name", 
      label: "Skills", 
      sortable: true,
      clickable: true,
      nameKey: "skills_name"
    },
    "Mobile": { key: "mobile_number", label: "Mobile", sortable: true },
    "Email": { key: "email_id", label: "Email", sortable: true },
    "Status": { key: "status", label: "Status", sortable: false },
    "Created Date": { key: "join_date", label: "Created Date", sortable: false },
  };

  const actionFormats = [
    { label: "view", modal: true, url: "/workforce/view/{id}" },
    { label: "edit", modal: true, url: "/workforce/update/{id}" },
    { label: "delete", modal: true, url: "/workforce/delete/{id}" },
    {
      label: "assign",
      modal: true,
      url: "/workforce/assign/{id}",
      showIf: (row) => row.is_fleet_assigned === false
    }
  ];

  const handleActionClick = (workforceId, action) => {
    setSelectedWorkforce({
      workforceId,
      modalType: action.label === "view"
        ? "view"
        : action.label === "edit"
        ? "update"
        : action.label === "assign"
        ? "assign"
        : "delete"
    });
  };

  const handleCellClick = (header, item) => {
    const mappingValue = WORKFORCE_HEADER_MAP[header];
    
    if (!mappingValue || !mappingValue.clickable) {
      return;
    }

    console.log('Cell clicked:', header, item);

    switch (mappingValue.key) {
      case 'first_name':
        const workforceId = item.workforce_id;
        if (workforceId) {
          setSelectedWorkforce({
            workforceId,
            modalType: 'view',
            data: item
          });
        }
        break;

      case 'workforce_category_name':
        const categoryId = item[mappingValue.idKey];
        if (categoryId) {
          setSelectedCategory({
            workforceCategoryId: categoryId,
            modalType: 'view',
            data: item
          });
        } else {
          console.warn('Workforce category ID not found in item:', item);
        }
        break;

      case 'skills_name':
        setSelectedSkill({
          name: item[mappingValue.nameKey],
          modalType: 'view',
          data: item
        });
        break;

      default:
        console.log('No special action for this field');
    }
  };

  const closeModal = () => {
    setSelectedWorkforce(null);
    setSelectedCategory(null);
    setSelectedSkill(null);
  };

  const handleSearch = (params) => {
    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      status: params.status || '',
      join_date_from: params.join_date_from || '',
      join_date_to: params.join_date_to || '',
      gender: params.gender || '',
      category: params.category || '',
      skip: 0
    }));
  };

  const handlePageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage
    }));
  };

  const buildFetchUrl = () => {
    let url = `/workforce/list/?skip=${searchParams.skip}&limit=${searchParams.limit}`;
    if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
    if (searchParams.status) url += `&status=${searchParams.status}`;
    if (searchParams.join_date_from) url += `&join_date_from=${searchParams.join_date_from}`;
    if (searchParams.join_date_to) url += `&join_date_to=${searchParams.join_date_to}`;
    if (searchParams.gender) url += `&gender=${searchParams.gender}`;
    if (searchParams.category) url += `&category=${searchParams.category}`;
    
    // Add account_id filter if provided
    if (accountId) {
      url += `&account_id=${accountId}`;
    }
    
    return url;
  };

  const scrollbarStyles = `
    ::-webkit-scrollbar {
      width: 8px;
      height: 8px;
    }
  `;

  return (
    <div className="workforce-list" style={{ overflow: 'hidden' }}>
      <style>{scrollbarStyles}</style>
      <SearchPage
        fetchDataUrl={buildFetchUrl()}
        ListComponent={List}
        itemKeyMapping={WORKFORCE_HEADER_MAP}
        dataKey="workforce"
        identifierKey="workforce_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}
        searchFields={searchFields}
        onSearch={handleSearch}
        onPageChange={handlePageChange}
      />

      {/* Workforce View Modal */}
      {selectedWorkforce?.modalType === "view" && (
        <WorkforceView
          workforceId={selectedWorkforce.workforceId}
          workforceData={selectedWorkforce.data}
          onClose={closeModal}
        />
      )}

      {/* Assignment Modal */}
      {selectedWorkforce?.modalType === "assign" && (
        <FleetAssignmentForm
          workforceId={selectedWorkforce.workforceId}
          onClose={closeModal}
        />
      )}

      {/* Update Modal */}
      {selectedWorkforce?.modalType === "update" && (
        <WorkforceUpdate
          workforceId={selectedWorkforce.workforceId}
          onClose={closeModal}
        />
      )}

      {/* Delete Modal */}
      {selectedWorkforce?.modalType === "delete" && (
        <WorkforceDB
          workforceId={selectedWorkforce.workforceId}
          onClose={closeModal}
        />
      )}

      {/* Workforce Category Modal */}
      {selectedCategory && (
        <div className="modal-overlay" style={{ overflow: 'auto' }}>
          <div className="modal-content">
            <WorkforceCategoryView
              workforceCategoryId={selectedCategory.workforceCategoryId}
              onClose={closeModal}
            />
          </div>
        </div>
      )}

      {/* Skills Modal */}
      {selectedSkill && (
        <div className="modal-overlay" style={{ overflow: 'auto' }}>
          <div className="modal-content" style={{ marginLeft: '60px' }}>
            <Skills
              skillName={selectedSkill.name}
              skillData={selectedSkill.data}
              onClose={closeModal}
            />
          </div>
        </div>
      )}
    </div>
  );
};

export default WorkforceList;