import Form from '@rjsf/core';
import React, { useState, useEffect } from 'react';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';

// Custom Date Widget Component
const CustomDateWidget = (props) => {
  const { value, onChange, options } = props;
  const today = new Date().toISOString().split('T')[0];

  return (
    <input
      type="date"
      className="form-control"
      value={value || ''}
      onChange={(e) => onChange(e.target.value)}
      max={options.maxDate || undefined}
      min={options.minDate || undefined}
    />
  );
};

// Custom File Widget Component (unchanged from your original)
const CustomFileWidget = (props) => {
    const { value, onChange } = props;

    const handleChange = (e) => {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = (event) => {
                onChange(event.target.result);
            };
            reader.readAsDataURL(file);
        }
    };

    return (
        <div style={{ margin: '20px 0' }}>
            {value && (
                <div style={{ marginBottom: '10px' }}>
                    <img 
                        src={value} 
                        alt="Preview" 
                        style={{ 
                            maxWidth: '150px', 
                            maxHeight: '150px',
                            borderRadius: '4px',
                            border: '1px solid #ddd'
                        }} 
                    />
                </div>
            )}
            <input 
                type="file" 
                onChange={handleChange} 
                accept="image/*"
                style={{ display: 'block' }}
            />
            <p style={{ fontSize: '0.8rem', color: '#666', marginTop: '5px' }}>
                Upload a clear photo of the workforce
            </p>
        </div>
    );
};

export const workforceSchema = {
    type: "object",
    required: ["first_name", "last_name", "mobile_number", "email_id", "gender", "workforce_category_id", "photo"],
    properties: {
        first_name: { type: "string", title: "First Name", minLength: 2, maxLength: 50 },
        last_name: { type: "string", title: "Last Name", minLength: 2, maxLength: 50 },
        mobile_number: { 
            type: "string", 
            title: "Mobile Number",
            pattern: "^[0-9]{10,15}$"
        },
        email_id: { 
            type: "string", 
            format: "email", 
            title: "Email",
            maxLength: 100
        },
        workforce_category_id: {
            type: "string",
            title: "Workforce Category",
            enum: [],
            enumNames: []
        },
        gender: { 
            type: "string", 
            title: "Gender",
            enum: ["Male", "Female", "Other", "Prefer not to say"]
        },
        date_of_birth: { 
            type: "string", 
            format: "date", 
            title: "Date of Birth" 
        },
        license_id: { 
            type: "string", 
            title: "License ID",
            maxLength: 50
        },
        national_id: { 
            type: "string", 
            title: "National ID",
            maxLength: 50
        },
        emergency_contact: { 
            type: "string", 
            title: "Emergency Contact",
            pattern: "^[0-9]{10,15}$"
        },
        shift_timing: { 
            type: "string", 
            title: "Shift Timing",
            enum: ["Morning", "Afternoon", "Evening", "Night", "Flexible"]
        },
        address: { 
            type: "string", 
            title: "Address",
            maxLength: 200
        },
        blood_group: { 
            type: "string", 
            title: "Blood Group",
            enum: ["A+", "A-", "B+", "B-", "AB+", "AB-", "O+", "O-", "Unknown"]
        },
        is_human: { 
            type: "boolean", 
            title: "Is Human", 
            default: true 
        },
        license_number: { 
            type: "string", 
            title: "License Number",
            maxLength: 50
        },
        join_date: { 
            type: "string", 
            format: "date", 
            title: "Join Date" 
        },
        photo: { 
            type: "string", 
            title: "Photo",
            format: "data-url"
        },
        skills_id: {
            type: "string",
            title: "Primary Skill",
            enum: [],
            enumNames: []
        }
    }
};

export const workforceUiSchema = {
    "ui:options": {
        submitButtonOptions: {
            submitText: "Submit",
            norender: false,
            props: {
                disabled: false,
                className: "btn btn-success"
            }
        }
    },
    first_name: {
        "ui:placeholder": "Enter first name",
        "ui:autofocus": true
    },
    last_name: {
        "ui:placeholder": "Enter last name"
    },
    mobile_number: {
        "ui:placeholder": "Enter 10-15 digit number",
        "ui:options": {
            inputType: "tel"
        }
    },
    email_id: {
        "ui:placeholder": "Enter valid email address",
        "ui:options": {
            inputType: "email"
        }
    },
    workforce_category_id: {
        "ui:widget": "select",
        "ui:placeholder": "Select category",
        "ui:options": {
            label: false
        }
    },
    gender: { 
        "ui:widget": "select",
        "ui:placeholder": "Select gender"
    },
    blood_group: { 
        "ui:widget": "select",
        "ui:placeholder": "Select blood group"
    },
    date_of_birth: { 
        "ui:widget": CustomDateWidget,
        "ui:options": {
            maxDate: new Date().toISOString().split('T')[0] // No future dates
        }
    },
    join_date: { 
        "ui:widget": CustomDateWidget,
        "ui:options": {
            minDate: new Date().toISOString().split('T')[0] // No past dates
        }
    },
    shift_timing: {
        "ui:widget": "select",
        "ui:placeholder": "Select shift"
    },
    is_human: { 
        "ui:widget": "checkbox",
        "ui:options": {
            label: true
        }
    },
    address: {
        "ui:widget": "textarea",
        "ui:options": {
            rows: 3
        }
    },
    photo: {
        "ui:widget": "fileWidget",
        "ui:options": {
            accept: "image/*"
        }
    },
    skills_id: {
        "ui:widget": "select",
        "ui:placeholder": "Select primary skill",
        "ui:options": {
            classNames: "custom-skill-spacing"
        }
    }
};

const WorkforceForm = ({ initialData, onSubmit, schema, uiSchema }) => {
    const [formData, setFormData] = useState(initialData || {});
    const [skillsList, setSkillsList] = useState([]);
    const [loading, setLoading] = useState(true);
    const [categories, setCategories] = useState([]);

    useEffect(() => {
        setFormData(initialData || {});
        const fetchInitialData = async () => {
            try {
                // Fetch skills
                const skillsResponse = await axios.get('crud/skills?skip=0&limit=100&q=&status=');
                setSkillsList(skillsResponse.data.items.map(skill => ({
                    id: skill.skills_id.toString(),
                    name: skill.name
                })));

                // Fetch categories
                const categoriesResponse = await axios.get('workforcecategory/list/');
                setCategories(categoriesResponse.data.workforce_categories.map(cat => ({
                    id: cat.workforcecategory_id,
                    name: cat.category_name
                })));
            } catch (error) {
                console.error("Error fetching initial data:", error);
            } finally {
                setLoading(false);
            }
        };
        fetchInitialData();
    }, [initialData]);

    const handleSubmit = ({ formData }) => {
        // Map IDs to names
        const selectedCategory = categories.find(
            cat => cat.id === formData.workforce_category_id
        );
        const selectedSkill = skillsList.find(
            skill => skill.id === formData.skills_id
        );

        const submissionData = {
            ...formData,
            workforce_category_name: selectedCategory?.name || '',
            skills_name: selectedSkill?.name || '',
            date_of_birth: formData.date_of_birth 
                ? new Date(formData.date_of_birth).toISOString() 
                : null,
            join_date: formData.join_date 
                ? new Date(formData.join_date).toISOString() 
                : null,
        };

        onSubmit(submissionData);
    };

    const dynamicSchema = {
        ...(schema || workforceSchema),
        properties: {
            ...(schema || workforceSchema).properties,
            workforce_category_id: {
                ...(schema || workforceSchema).properties.workforce_category_id,
                enum: categories.map(cat => cat.id),
                enumNames: categories.map(cat => cat.name)
            },
            skills_id: {
                ...(schema || workforceSchema).properties.skills_id,
                enum: skillsList.map(skill => skill.id),
                enumNames: skillsList.map(skill => skill.name)
            }
        }
    };

    const widgets = {
        fileWidget: CustomFileWidget,
        CustomDateWidget: CustomDateWidget
    };

    if (loading) {
        return <div style={{ textAlign: 'center', padding: '20px' }}>Loading form data...</div>;
    }

    return (
        <div style={{ maxWidth: '600px', padding: '20px' }}>
            <style>
                {`
                    .custom-skill-spacing {
                        margin-bottom: 19px;
                    }
                `}
            </style>
            <Form
                schema={dynamicSchema}
                uiSchema={uiSchema || workforceUiSchema}
                formData={formData}
                onChange={({ formData }) => setFormData(formData)}
                onSubmit={handleSubmit}
                validator={validator}
                widgets={widgets}
                templates={{
                    ButtonTemplates: {
                        submitBtn: (props) => (
                            <button 
                                {...props} 
                                className="btn btn-success"
                                style={{
                                    padding: '10px 20px',
                                    backgroundColor: '#4CAF50',
                                    color: 'white',
                                    border: 'none',
                                    borderRadius: '1px',
                                    cursor: 'pointer',
                                    fontSize: '12px',
                                    fontWeight: 'bold',
                                    transition: 'background-color 0.3s',
                                    ':hover': {
                                        backgroundColor: '#45a049'
                                    }
                                }}
                            >
                                Submit
                            </button>
                        )
                    }
                }}
            />
        </div>
    );
};

export default WorkforceForm;