import React, { useEffect, useState } from 'react';
import axios from 'axios';
import '../../themes/blue/Workforceview.css';

const FleetAssignmentForm = ({ onClose, workforceId }) => {
  const [searchQuery, setSearchQuery] = useState('');
  const [filteredFleets, setFilteredFleets] = useState([]);
  const [selectedFleetId, setSelectedFleetId] = useState('');
  const [assignFrom, setAssignFrom] = useState('');
  const [assignTo, setAssignTo] = useState('');
  const [isPermanent, setIsPermanent] = useState(false);
  const [loading, setLoading] = useState(false);
  const [accountId, setAccountId] = useState('');
  const [userId, setUserId] = useState('');
  const [minDate, setMinDate] = useState(''); // ✅ New state for minimum date

  useEffect(() => {
    const tokenData = localStorage.getItem('token');
    if (tokenData) {
      try {
        const userData = JSON.parse(atob(tokenData.split('.')[1]));
        setAccountId(userData.account_id || '');
        setUserId(userData.user_id || '');
      } catch (error) {
        console.error('Error parsing token:', error);
      }
    }

    // ✅ Set default "assignFrom" to current datetime (rounded to nearest minute)
    const now = new Date();
    now.setSeconds(0, 0);
    const offset = now.getTimezoneOffset() * 60000;
    const localISOTime = new Date(now - offset).toISOString().slice(0, 16);
    setAssignFrom(localISOTime);
    setMinDate(localISOTime); // ✅ Set min date for validation
  }, []);

  useEffect(() => {
    if (assignFrom) {
      const fromDate = new Date(assignFrom);
      const toDate = new Date(fromDate);
      toDate.setDate(toDate.getDate() + 1);
      const offset = toDate.getTimezoneOffset() * 60000;
      const localISOTime = new Date(toDate - offset).toISOString().slice(0, 16);
      setAssignTo(localISOTime);
    }
  }, [assignFrom]);

  useEffect(() => {
    const delayDebounceFn = setTimeout(() => {
      if (searchQuery.trim().length >= 3) {
        fetchFleets(searchQuery.trim());
      } else {
        setFilteredFleets([]);
        setSelectedFleetId('');
      }
    }, 300);
    return () => clearTimeout(delayDebounceFn);
  }, [searchQuery]);

  const fetchFleets = async (query) => {
    try {
      const token = localStorage.getItem('token');
      const res = await axios.get(
        `fleets/list/all/?skip=0&limit=10&q=${encodeURIComponent(query)}`,
        {
          headers: { Authorization: `Bearer ${token}` },
        }
      );
      const fleets = res.data.fleets || [];
      setFilteredFleets(fleets);
    } catch (err) {
      console.error('Error fetching fleets:', err);
      setFilteredFleets([]);
      alert('Error fetching fleet data');
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    if (!selectedFleetId || !assignFrom || !assignTo) {
      alert('Please enter a valid fleet and both date fields are required.');
      return;
    }

    const payload = {
      account_id: localStorage.getItem('account_id'),
      user_id: localStorage.getItem('user_id'),
      workforce_id: workforceId,
      vehicle_id: selectedFleetId,
      assigned_from: new Date(assignFrom).toISOString(),
      assigned_to: new Date(assignTo).toISOString(),
      status: true,
      is_permanent: isPermanent,
    };

    setLoading(true);
    try {
      const token = localStorage.getItem('token');
      await axios.post('workforcevehicle/', payload, {
        headers: {
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`,
        },
      });
      alert(`Fleet assigned successfully!`);
      onClose(true);
    } catch (err) {
      console.error('Assignment failed:', err);
      alert(err.response?.data?.message || 'Assignment failed');
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      <style>{`
        .assign-overlay {
          position: fixed;
          top: 0;
          left: 0;
          width: 100vw;
          height: 100vh;
          background: rgba(0, 0, 0, 0.3);
          z-index: 999;
        }

        .assign-drawer {
          position: fixed;
          top: 0;
          right: 0;
          width: clamp(320px, 50vw, 900px);
          padding: 30px;
          height: 100%;
          background: white;
          box-shadow: -2px 0 8px rgba(0, 0, 0, 0.1);
          z-index: 1000;
          display: flex;
          flex-direction: column;
          transition: width 0.3s ease;
        }

        .assign-drawer-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          margin-bottom: 20px;
        }

        .assign-close-btn {
          font-size: 24px;
          background: none;
          border: none;
          cursor: pointer;
        }

        .form-group {
          margin-bottom: 16px;
          position: relative;
        }

        input[type="text"], input[type="datetime-local"] {
          width: 100%;
          padding: 8px;
          font-size: 16px;
          border: 1px solid #ccc;
          border-radius: 4px;
        }

        .assign-btn {
          padding: 10px 16px;
          background-color: #f0f0f0;
          color:	#4CAF50;
          font-size: 14px;
          border: 1px solid white;
          border-radius: 4px;
          cursor: pointer;
          transition: all 0.3s ease;
        }

        .assign-btn:hover {
          background-color: #4CAF50;
          color: white;
        }

        .assign-btn:disabled {
          opacity: 0.6;
          cursor: not-allowed;
        }

        .radio-group {
          display: flex;
          gap: 20px;
        }

        .fleet-suggestions {
          margin-top: 5px;
          list-style: none;
          padding: 0;
          border: 1px solid #ccc;
          max-height: 150px;
          overflow-y: auto;
          background: #fff;
          position: absolute;
          z-index: 9999;
          width: 100%;
        }

        .fleet-suggestions li {
          padding: 8px;
          cursor: pointer;
          border-bottom: 1px solid #eee;
        }

        .fleet-suggestions li:hover {
          background-color: #f0f0f0;
        }
      `}</style>

      <div className="assign-overlay" onClick={onClose} />
      <div className="assign-drawer" onClick={(e) => e.stopPropagation()}>
        <div className="assign-drawer-header">
          <h2>Assign Fleet</h2>
          <button onClick={onClose} className="assign-close-btn">&times;</button>
        </div>

        <div className="assign-drawer-body">
          <form onSubmit={handleSubmit}>
            <div className="form-group">
              <label>Search Fleet Number:</label>
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Type fleet number (min 3 characters)"
                disabled={loading}
              />
              {filteredFleets.length > 0 && (
                <ul className="fleet-suggestions">
                  {filteredFleets.map((fleet) => (
                    <li
                      key={fleet.fleet_id}
                      onClick={() => {
                        setSearchQuery(fleet.fleet_name);
                        setSelectedFleetId(fleet.fleet_id);
                        setFilteredFleets([]);
                      }}
                    >
                      {fleet.fleet_name}
                    </li>
                  ))}
                </ul>
              )}
            </div>

            <div className="form-group">
              <label>Assign From:</label>
              <input
                type="datetime-local"
                value={assignFrom}
                onChange={(e) => setAssignFrom(e.target.value)}
                required
                min={minDate} // ✅ Prevent past dates
              />
            </div>

            <div className="form-group">
              <label>Assign To:</label>
              <input
                type="datetime-local"
                value={assignTo}
                onChange={(e) => setAssignTo(e.target.value)}
                required
                min={assignFrom} // ✅ Prevent 'to' date before 'from'
              />
            </div>

            <div className="form-group">
              <label>Is Permanent:</label>
              <div className="radio-group">
                <label>
                  <input
                    type="radio"
                    checked={isPermanent === true}
                    onChange={() => setIsPermanent(true)}
                  />
                  Yes
                </label>
                <label>
                  <input
                    type="radio"
                    checked={isPermanent === false}
                    onChange={() => setIsPermanent(false)}
                  />
                  No
                </label>
              </div>
            </div>

            <button 
              type="submit" 
              className="assign-btn"
              disabled={loading}
            >
              {loading ? 'Assigning...' : 'Assign Fleet'}
            </button>
          </form>
        </div>
      </div>
    </>
  );
};

export default FleetAssignmentForm;
