import React, { useEffect, useState } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import WorkforceForm from './WorkforceForm';
import { workforceSchema, workforceUiSchema } from './WorkforceForm';

// Define scrollbar styles as a constant
const scrollbarStyles = `
  ::-webkit-scrollbar {
    width: 8px;
    height: 8px;
  }
//   ::-webkit-scrollbar-track {
//     background: #f1f1f1;
//   }
//   ::-webkit-scrollbar-thumb {
//     background: #888;
//     border-radius: 4px;
//   }
  ::-webkit-scrollbar-thumb:hover {
    background: #555;
  }
`;


const dataURLtoFile = (dataurl, filename) => {
  const arr = dataurl.split(',');
  const mime = arr[0].match(/:(.*?);/)[1];
  const bstr = atob(arr[1]);
  let n = bstr.length;
  const u8arr = new Uint8Array(n);
  while (n--) {
    u8arr[n] = bstr.charCodeAt(n);
  }
  return new File([u8arr], filename, { type: mime });
};

const NewWorkforce = ({ onClose, onCreateSuccess }) => {
    const [categories, setCategories] = useState([]);
    const [loadingCategories, setLoadingCategories] = useState(false);
    const [categoriesError, setCategoriesError] = useState('');

    useEffect(() => {
        const fetchCategories = async () => {
            try {
                setLoadingCategories(true);
                const response = await axios.get('/workforcecategory/list/');
                const categoriesData = response.data?.workforce_categories || [];
                setCategories(Array.isArray(categoriesData) ? categoriesData : []);
            } catch (err) {
                setCategoriesError('Failed to load workforce categories');
                console.error('Category fetch error:', err);
            } finally {
                setLoadingCategories(false);
            }
        };
        fetchCategories();
    }, []);

    const categoryOptions = Array.isArray(categories) ? categories : [];
    
    const modifiedSchema = {
        ...workforceSchema,
        properties: {
            ...workforceSchema.properties,
            workforce_category_id: {
                type: "string",
                title: "Workforce Category",
                enum: categoryOptions.map(cat => cat.workforcecategory_id),
                enumNames: categoryOptions.map(cat => cat.category_name),
            }
        },
        required: [...workforceSchema.required, "workforce_category_id"]
    };

    const modifiedUiSchema = {
        ...workforceUiSchema,
        workforce_category_id: {
            "ui:widget": "select",
            "ui:placeholder": "Select a category"
        }
    };

    const handleSubmit = (formData) => {
      const formDataToSend = new FormData();
      
      // Append all fields except 'photo'
      Object.keys(formData).forEach(key => {
          if (key !== 'photo') {
              formDataToSend.append(key, formData[key]);
          }
      });

      // Handle photo upload if exists
      if (formData.photo && formData.photo.startsWith('data:')) {
          const photoFile = dataURLtoFile(formData.photo, `workforce-${Date.now()}.png`);
          formDataToSend.append('photo', photoFile);
      }

      // Append account/user context
      formDataToSend.append('account_id', localStorage.getItem('account_id'));
      formDataToSend.append('user_id', localStorage.getItem('user_id'));

      axios.post('/workforce/', formDataToSend, {
          headers: {
              'Content-Type': 'multipart/form-data'
          }
      })
      .then(res => {
          alert("Workforce created successfully!");
          onCreateSuccess?.(res.data);
          onClose?.();
      })
      .catch(err => {
          console.error('Error:', err.response?.data);
          alert("Error creating workforce: " + 
              (err.response?.data?.detail || "Unknown error"));
      });
  };

  
    
return (
  <div style={{ display: 'flex', padding: '2px', justifyContent: 'flex-start' }}> {/* Changed to flex-start */}
     <style>{scrollbarStyles}</style>
      {categoriesError && <div className="error">{categoriesError}</div>}
      {loadingCategories ? (
          <div>Loading categories...</div>
      ) : categories.length > 0 ? (
        <div className="job-form-container" style={{ maxWidth: '600px', width: '100%', marginLeft: 0 }}> {/* Added marginLeft: 0 */}
          <WorkforceForm 
              onSubmit={handleSubmit}
              schema={modifiedSchema}
              uiSchema={modifiedUiSchema}
              key={categories.length}
          />
          </div>
      ) : (
          <div>No categories available</div>
      )}
  </div>
);
};

export default NewWorkforce;
