import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import CommonTaskList from '../../common/CommonTaskList';
import TaskBar from '../../common/TaskListBar';
import '../../themes/blue/Workforceview.css';

const TAB_CONFIG = [
  { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
  { id: 'tasks', label: 'Tasks', icon: 'assignment' },
  { id: 'contracts', label: 'Contracts', icon: 'description' },
  { id: 'services', label: 'Services', icon: 'build' },
  { id: 'documents', label: 'Documents', icon: 'folder' },
  { id: 'info', label: 'Info', icon: 'info' }
];

const VendorView = ({ vendorId, vendorData: initialData, onClose }) => {
  const [vendorData, setVendorData] = useState(initialData || null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');
  const [hoveredTab, setHoveredTab] = useState(null);


    useEffect(() => {
    if (!initialData) {
      axios.get(`/vendors/${vendorId}`)
        .then(res => setVendorData(res.data))
        .catch(() => setError('Error loading vendor details'))
        .finally(() => setLoading(false));
    } else {
      setLoading(false);
    }
  }, [vendorId]);






  const renderTabContent = () => {
    if (!vendorData) return null;

    switch (activeTab) {
      case 'dashboard':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>dashboard</span>
              Vendor Summary
            </h4>
            <p><strong>Name:</strong> {vendorData.vendor_name}</p>
            <p><strong>Type:</strong> {vendorData.vendor_type}</p>
            <p><strong>Status:</strong> {vendorData.status ? 'Active' : 'Inactive'}</p>
            {/* <p><strong>Since:</strong> {new Date(vendorData.registration_date).toLocaleDateString()}</p> */}

            <TaskBar vendorId={vendorId} />

            <h4 style={{ marginTop: '1rem' }}>Recent Activity</h4>
            <p><strong>Last 2 Contracts:</strong> (placeholder)</p>
            <p><strong>Last Service:</strong> {vendorData.last_service_date || '-'}</p>
          </div>
        );

      case 'tasks':
        return (
          <div className="tab-content">
            <h4><span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>assignment</span>Vendor Tasks</h4>
            <div className="task-list-container">
              <CommonTaskList
                presetFilters={{ vendor_id: vendorId }}
                rowsPerPage={5}
                initialSearchQuery=""
              />
            </div>
          </div>
        );

      case 'contracts':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>description</span>
              Contracts
            </h4>
            <div className="contracts-grid">
              <div className="contract-card">
                <h5>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>receipt</span>
                  Current Contract
                </h5>
                <p><strong>Contract ID:</strong> {vendorData.current_contract?.id || '--'}</p>
                <p><strong>Start Date:</strong> {vendorData.current_contract?.start_date || '--'}</p>
                <p><strong>End Date:</strong> {vendorData.current_contract?.end_date || '--'}</p>
                <p><strong>Value:</strong> {vendorData.current_contract?.value || '--'}</p>
              </div>
              <div className="contract-card">
                <h5>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>history</span>
                  Contract History
                </h5>
                <p>Contract A (Jan 2022 - Dec 2022)</p>
                <p>Contract B (Jan 2021 - Dec 2021)</p>
              </div>
            </div>
          </div>
        );

      case 'services':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>build</span>
              Services Provided
            </h4>
            <div className="services-list">
              {vendorData.services?.length > 0 ? (
                vendorData.services.map(service => (
                  <div key={service.id} className="service-item">
                    <p><strong>{service.name}</strong> - {service.description}</p>
                    <p>Rate: {service.rate}</p>
                  </div>
                ))
              ) : (
                <p>No services registered</p>
              )}
            </div>
          </div>
        );

      case 'documents':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>folder</span>
              Vendor Documents
            </h4>
            <div className="document-grid">
              <div className="detail-row"><strong>License Number:</strong> {vendorData.license_number || '--'}</div>
              <div className="detail-row"><strong>License Expiry:</strong> {vendorData.license_expiry || '--'}</div>
              <div className="detail-row"><strong>Tax ID:</strong> {vendorData.tax_id || '--'}</div>
              <div className="detail-row"><strong>Insurance:</strong> {vendorData.insurance_number || '--'}</div>
              <div className="detail-row"><strong>Quality Certificates:</strong> {vendorData.quality_certificates || '--'}</div>
            </div>
          </div>
        );

      case 'info':
        return (
          <div className="tab-content">
            <div className="profile-grid">
              <div className="profile-column">
                <h4>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>info</span>
                  Vendor Information
                </h4>
                <div className="detail-row"><strong>Vendor Name:</strong> {vendorData.vendor_name}</div>
                <div className="detail-row"><strong>Type:</strong> {vendorData.vendor_type}</div>
                <div className="detail-row"><strong>Contact Person:</strong> {vendorData.contact_person_name}</div>
                <div className="detail-row"><strong>Email:</strong> {vendorData.email}</div>
                <div className="detail-row"><strong>Phone:</strong> {vendorData.phone}</div>
                <div className="detail-row"><strong>Website:</strong> {vendorData.website || '--'}</div>
                <div className="detail-row"><strong>Registration Date:</strong> {new Date(vendorData.registration_date).toLocaleDateString()}</div>
              </div>
              <div className="profile-column">
                <h4>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>location_on</span>
                  Address Details
                </h4>
                <div className="detail-row"><strong>Street:</strong> {vendorData.address?.street || '--'}</div>
                <div className="detail-row"><strong>City:</strong> {vendorData.address?.city || '--'}</div>
                <div className="detail-row"><strong>State:</strong> {vendorData.address?.state || '--'}</div>
                <div className="detail-row"><strong>Postal Code:</strong> {vendorData.address?.postal_code || '--'}</div>
                <div className="detail-row"><strong>Country:</strong> {vendorData.address?.country || '--'}</div>
              </div>
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading vendor data...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }

  return (
    <Modal onClose={onClose}>
      <div className="fixed top-0 right-0 w-full md:w-[55%] lg:w-[45%] xl:w-[40%] h-full bg-white shadow-lg z-50 overflow-hidden">
        {/* Header Section */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          padding: '10px 10px 20px',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px'
        }}>
          <div style={{ display: 'flex', alignItems: 'center' }}>
            <div style={{
              width: '60px',
              height: '60px',
              borderRadius: '50%',
              backgroundColor: '#e0e0e0',
              color: '#000',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              fontSize: '20px',
              fontWeight: 'bold',
              marginRight: '16px'
            }}>
              {vendorData.vendor_name?.charAt(0).toUpperCase()}
            </div>
            <div>
              <div style={{
                fontWeight: '600',
                fontSize: '20px',
                lineHeight: '1.4',
                color: 'black'
              }}>
                {vendorData.vendor_name}
              </div>
              <div style={{
                fontSize: '16px',
                color: 'black',
                marginTop: '10px',
                fontWeight: 'bold'
              }}>
                ID: #{vendorData.v_id}
              </div>
            </div>
          </div>
        </div>

        {/* Tabs Section */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '0px 10px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '20px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px',
                transition: 'min-width 0.2s ease'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && (
                <span>{tab.label}</span>
              )}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-1px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Tab Content */}
        <div style={{ maxHeight: 'calc(100vh - 250px)', overflowY: 'auto', padding: '0 20px' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default VendorView;