


import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import VendorView from './VendorView';
import VendorUpdate from './VendorUpdate';
import VendorDelete from './VendorDelete';
import CommonData from '../../common/CommonData';

const VENDOR_HEADER_TO_KEY_MAP = {
  "Vendor Name": { 
    key: "vendor_name", 
    label: "Vendor Name", 
    sortable: true,
    clickable: true  // Added clickable property
  },
  "Vendor Type": { key: "vendor_type", label: "Vendor Type", sortable: true },
  "Contact": { key: "contact_person_name", label: "Contact", sortable: true },
  "Phone": { key: "phone", label: "Phone", sortable: true },
  "Email": { key: "email", label: "Email", sortable: true },
  "Created Date": { key: "created_date", label: "Created Date", sortable: false },
  "Status": { key: "status", label: "Status", sortable: false },
};

const VendorsList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const fetchDataUrl = `/vendors/list/all/`;
  const [selectedVendor, setSelectedVendor] = useState(null);
  const [selectedVendorFromCell, setSelectedVendorFromCell] = useState(null);
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  
  // Get common enum values for dropdowns
  const enums = CommonData();

  const actionFormats = [
    { label: "view", modal: true, url: "/vendors/view/{id}" },
    { label: "edit", modal: true, url: "/vendors/update/{id}" },
    { label: "delete", modal: true, url: "/vendors/delete/{id}" },
  ];

  // Vendor-specific search and filter fields
  const searchFields = [
    { 
      name: 'date_from', 
      type: 'date', 
      placeholder: 'From Date' 
    },
    { 
      name: 'date_to', 
      type: 'date', 
      placeholder: 'To Date' 
    },
        { 
      name: 'q', 
      type: 'text', 
      placeholder: 'Search by vendor name, email...', 
      defaultValue: initialSearchQuery 
    },
      {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All Statuses', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' },
        { label: 'Blocked', value: 'blocked' },
      ],
    },
    {
      name: 'vendor_type',
      type: 'dropdown',
      placeholder: 'Vendor Type',
      options: [
        { label: 'All Types', value: '' },
        ...(enums.VendorType || []).map(val => ({ 
          label: val, 
          value: val.toLowerCase() 
        })),
      ],
    },
  
  ];

  const handleActionClick = (vendorId, action) => {
    if (action.label === "view") {
      setSelectedVendor({ vendorId, modalType: "view" });
    } else if (action.label === "edit") {
      setSelectedVendor({ vendorId, modalType: "update" });
    } else if (action.label === "delete") {
      setSelectedVendor({ vendorId, modalType: "delete" });
    }
  };

  const handleCellClick = (header, item) => {
    if (header === "Vendor Name") {
      console.log('Vendor Name clicked:', item.vendor_id);
      setSelectedVendorFromCell({
        vendorId: item.vendor_id,
        modalType: "view"
      });
    }
  };

  const closeModal = () => {
    setSelectedVendor(null);
    setSelectedVendorFromCell(null);
  };

  const refreshList = () => setRefreshTrigger(prev => prev + 1);

  return (
    <div>
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={VENDOR_HEADER_TO_KEY_MAP}
        dataKey="vendors"
        identifierKey="vendor_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}  // Added onCellClick handler
        refreshTrigger={refreshTrigger}
        searchFields={searchFields}
      />

      {/* Modal for actions (view/edit/delete) */}
      {selectedVendor?.modalType === "view" && (
        <VendorView 
          vendorId={selectedVendor.vendorId}
          onClose={closeModal}
        />
      )}

      {selectedVendor?.modalType === "update" && (
        <VendorUpdate 
          vendorId={selectedVendor.vendorId}
          onClose={closeModal}
          onUpdateSuccess={refreshList}
        />
      )}

      {selectedVendor?.modalType === "delete" && (
        <VendorDelete 
          vendorId={selectedVendor.vendorId}
          onClose={closeModal}
          onDeleteSuccess={refreshList}
        />
      )}

      {/* Modal for Vendor Name click */}
      {selectedVendorFromCell?.modalType === "view" && (
        <VendorView 
          vendorId={selectedVendorFromCell.vendorId}
          onClose={closeModal}
        />
      )}
    </div>
  );
};

export default VendorsList;