

import React, { useState } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';

// Schema Definition
export const vendorSchema = {
  type: "object",
  required: [
    "vendor_name", 
    "vendor_type",
    "address",
    "phone",
    "contact_person_name",
    "email",
    "license_number"
  ],
  properties: {
    vendor_name: { 
      type: "string", 
      title: "Vendor Name", 
      minLength: 2, 
      maxLength: 100 
    },
    vendor_type: { 
      type: "string", 
      title: "Vendor Type",
      enum: ["vendor", "dealer"],
      default: "vendor"
    },
    address: {
      type: "object",
      title: "Address",
      required: ["street", "city", "state", "postal_code", "country"],
      properties: {
        street: { type: "string", title: "Street" },
        city: { type: "string", title: "City" },
        state: { type: "string", title: "State" },
        postal_code: { type: "string", title: "Postal Code" },
        country: { type: "string", title: "Country" }
      }
    },
    phone: { type: "string", title: "Phone Number" },
    contact_person_name: { type: "string", title: "Contact Person" },
    email: { type: "string", format: "email", title: "Email" },
    license_number: { type: "string", title: "License Number" },
    tax_id: { type: "string", title: "Tax ID (Optional)", default: "" },
    website: { type: "string", title: "Website (Optional)", default: "" },
    logo: { type: "string", title: "Vendor Logo (Optional)", default: "" }
  }
};

// UI Schema (with submit button enabled)
export const vendorUiSchema = {
  "ui:order": [
    "vendor_name",
    "vendor_type",
    "address",
    "phone",
    "contact_person_name",
    "email",
    "license_number",
    "tax_id",
    "website",
    "logo"
  ],
  "ui:submitButtonOptions": {
    norender: false,
    submitText: "Submit"
  },
  address: {
    "ui:title": "Address Details",
    "ui:options": {
      collapsible: true
    }
  },
  logo: {
    "ui:widget": "file"
  },
  website: {
    "ui:help": "Optional website URL"
  },
  tax_id: {
    "ui:help": "Optional tax identification number"
  }
};

// Main Component
const VendorForm = ({ initialData = {}, onSubmit }) => {
  const [formData, setFormData] = useState({
    tax_id: "",
    website: "",
    logo: "",
    ...initialData
  });
  const [uploading, setUploading] = useState(false);
  const [submitted, setSubmitted] = useState(false);

  const handleSubmit = async ({ formData }) => {
    setSubmitted(true);

    try {
      // If logo is uploaded (data-url), simulate upload to backend
      if (formData.logo && formData.logo.startsWith('data:')) {
        setUploading(true);
        const timestamp = Date.now();
        const base64Data = formData.logo.split(',')[1];
        const file = new File(
          [Uint8Array.from(atob(base64Data), c => c.charCodeAt(0))],
          `vendor-logo-${timestamp}.png`,
          { type: 'image/png' }
        );

        const logoForm = new FormData();
        logoForm.append('file', file);

        const response = await axios.post('/vendors', logoForm, {
          headers: { 'Content-Type': 'multipart/form-data' }
        });

        formData.logo = response.data.url; // Assume backend returns { url: "..." }
      }

      // Final form submit
      onSubmit(formData);
    } catch (err) {
      console.error("Error uploading or submitting form:", err);
      alert("Something went wrong while submitting the form.");
    } finally {
      setUploading(false);
    }
  };

  return (
    <div style={{ maxWidth: "600px", padding: "20px" }}>
       {/* Added style tag to push Submit button downward */}
  <style>
  {`
    form .form-group.field.field-object + .form-group {
      margin-top: 16px;
    }

    form .form-actions {
      margin-top: 40px; /* ✅ Increased space between last field and submit button */
    }

    /* ✅ Add margin-bottom to the last input (file upload) if needed */
    form .field-string input[type="file"] {
      margin-bottom: 18px;
    }
  `}
</style>
      <Form
        schema={vendorSchema}
        uiSchema={vendorUiSchema}
        formData={formData}
        onChange={({ formData }) => setFormData(formData)}
        onSubmit={handleSubmit}
        validator={validator}
        showErrorList={submitted}
        noHtml5Validate
        disabled={uploading}
      />
      {uploading && <p style={{ color: "blue" }}>Uploading logo, please wait...</p>}
    </div>
  );
};

export default VendorForm;







