import React, { useState, useEffect, useRef } from 'react';
import axios from 'axios';
import GoogleMapReact from 'google-map-react';
import Modal from '../../Layouts/Modal';

const getRandomColor = (str) => {
  let hash = 0;
  for (let i = 0; i < str.length; i++) {
    hash = str.charCodeAt(i) + ((hash << 5) - hash);
  }
  
  const colors = [
    '#F44336', '#E91E63', '#9C27B0', '#673AB7', '#3F51B5',
    '#2196F3', '#03A9F4', '#00BCD4', '#009688', '#4CAF50',
    '#8BC34A', '#CDDC39', '#FFC107', '#FF9800', '#FF5722',
    '#795548', '#607D8B'
  ];
  
  return colors[Math.abs(hash) % colors.length];
};

const getAvatarUrl = (name) => {
  const initials = name.charAt(0).toUpperCase();
  const backgroundColor = getRandomColor(name).replace('#', '');
  return `https://ui-avatars.com/api/?name=${initials}&background=${backgroundColor}&color=fff&size=128`;
};

const vehicles = [
  {
    id: '100001',
    number: 'TN 04 AB 4567',
    driver: 'John',
    address: 'Tidel Park, Peelamedu, Coimbatore',
    status: 'Scheduled',
    lastUpdated: '8:01 PM, 02 Jun, 2025',
    price: '$0',
    activities: [
      "Assigned to Driver4",
      "Scheduled for delivery",
      "Next maintenance due in 15 days"
    ],
    currentLocation: { lat: 11.0183, lng: 76.9725 }
  },
  {
    id: '100002',
    number: 'TN 05 AB 5678',
    driver: 'Liam',
    address: 'Vadavalli, Coimbatore',
    status: 'Available',
    lastUpdated: '5:30 AM, 01 Jan, 1970',
    price: '$0',
    activities: [
      "Ready for assignment",
      "Recently serviced",
      "Fuel level: 85%"
    ],
    currentLocation: { lat: 11.4083, lng: 76.6925 }
  },
  {
    id: '100003',
    number: 'TN 06 AB 6789',
    driver: 'William',
    address: 'B.R. Puram Industrial Estate',
    status: 'Available',
    lastUpdated: '5:30 AM, 01 Jan, 1970',
    price: '$0',
    activities: [
      "Ready for assignment",
      "Needs tire replacement",
      "Mileage: 45,000 km"
    ],
    currentLocation: { lat: 11.2183, lng: 76.8925 }
  },
  {
    id: '100004',
    number: 'TN 07 AB 7890',
    driver: 'Michael',
    address: 'Peelamedu, Coimbatore',
    status: 'In Maintenance',
    lastUpdated: '5:30 AM, 01 Jan, 1970',
    price: '$0',
    activities: [
      "Currently in service",
      "Engine check required",
      "Estimated completion: Tomorrow"
    ],
    currentLocation: { lat: 11.3183, lng: 76.7925 }
  },
  {
    id: '100005',
    number: 'TN 08 AB 8901',
    driver: 'Daniel',
    address: 'Gandhipuram, Coimbatore',
    status: 'LUX',
    lastUpdated: '5:30 AM, 01 Jan, 1970',
    price: '$120',
    activities: [
      "Premium vehicle",
      "Assigned to VIP client",
      "Next service in 30 days"
    ],
    currentLocation: { lat: 11.2583, lng: 76.9525 }
  }
];

const tasks = [
  {
    id: 'task001',
    driver: 'Jack',
    pickup: 'Tidel Park, Peelamedu, B.R. Puram Industrial Estate, Coimbatore, Tamil Nadu, India',
    dropoff: 'Vadavalli, Coimbatore, Tamil Nadu, India',
    pickupLocation: { lat: 11.0183, lng: 76.9725 },
    dropoffLocation: { lat: 11.0283, lng: 76.9825 },
    status: 'Scheduled',
    time: '8:01 PM, 02 Jun, 2025'
  },
  {
    id: 'task002',
    driver: 'Oliver',
    pickup: 'Race Course, Coimbatore, Tamil Nadu, India',
    dropoff: 'Saravanampatti, Coimbatore, Tamil Nadu, India',
    pickupLocation: { lat: 11.0083, lng: 76.9625 },
    dropoffLocation: { lat: 11.0383, lng: 76.9925 },
    status: 'In Progress',
    time: '9:30 AM, 03 Jun, 2025'
  },
  {
    id: 'task003',
    driver: 'Ramesh',
    pickup: 'Gandhipuram Bus Stand, Coimbatore, Tamil Nadu, India',
    dropoff: 'Coimbatore Airport, Peelamedu, Coimbatore, Tamil Nadu, India',
    pickupLocation: { lat: 11.0583, lng: 76.9425 },
    dropoffLocation: { lat: 11.0483, lng: 76.9725 },
    status: 'Completed',
    time: '3:45 PM, 01 Jun, 2025'
  }
];

const teams = [
  {
    id: 'team001',
    name: 'Logistics Team A',
    members: ['John', 'Jack', 'Oliver'],
    manager: 'Michael',
    status: 'Active'
  },
  {
    id: 'team002',
    name: 'VIP Transport Team',
    members: ['Daniel', 'William'],
    manager: 'Liam',
    status: 'Active'
  },
  {
    id: 'team003',
    name: 'Maintenance Crew',
    members: ['Ramesh', 'Kumar'],
    manager: 'Suresh',
    status: 'On Break'
  }
];

const TrackView = ({ setShowTabs }) => {
  const [socket, setSocket] = useState(null);
  const [isConnected, setIsConnected] = useState(false);
  const [realtimeMarkers, setRealtimeMarkers] = useState({});
  const [selectedVehicle, setSelectedVehicle] = useState(vehicles[0]);
  const [selectedTask, setSelectedTask] = useState(tasks[0]);
  const [selectedTeam, setSelectedTeam] = useState(teams[0]);
  const [isDetailModalOpen, setIsDetailModalOpen] = useState(false);
  const [trackData, setTrackData] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [userLocation, setUserLocation] = useState({ lat: 11.0183, lng: 76.9725 });
  const [isSidebarOpen, setIsSidebarOpen] = useState(true);
  const [rightSearch, setRightSearch] = useState('');
  const [activeTab, setActiveTab] = useState('vehicles');
  const [alerts, setAlerts] = useState([]);

  const mapRef = useRef(null);
  const [mapInstance, setMapInstance] = useState(null);
  const [mapsApi, setMapsApi] = useState(null);

  useEffect(() => {
    const ws = new WebSocket('ws://localhost:8003/ws/realtime');

    ws.onopen = () => {
      console.log('✅ Connected to WebSocket server');
      setIsConnected(true);
      setSocket(ws);
    };

    ws.onmessage = (event) => {
      try {
        const data = JSON.parse(event.data);
        
        // Handle alerts
        if (data.type === "alert") {
          showAlert(data.message || "Alert received");
          return;
        }

        const { device_id, location, speed, heading, received_at, alert } = data;

        if (alert) {
          showAlert(alert);
        }

        if (!device_id || !location?.coordinates) return;
        const [lng, lat] = location.coordinates;
        
        setRealtimeMarkers(prev => ({
          ...prev,
          [device_id]: {
            lat,
            lng,
            speed,
            heading,
            received_at,
          },
        }));
      } catch (e) {
        console.error('Failed to parse message:', e);
      }
    };

    ws.onerror = (err) => {
      console.error('❌ WebSocket error:', err);
    };

    ws.onclose = () => {
      console.warn('🔌 WebSocket connection closed');
      setIsConnected(false);
    };

    return () => ws.close();
  }, []);

  const showAlert = (msg) => {
    const timestamp = new Date().toLocaleTimeString();
    const newAlert = `${timestamp}: ${msg}`;
    setAlerts(prev => [newAlert, ...prev].slice(0, 20));
  };

  useEffect(() => {
    const header = document.querySelector('.PageHeader');
    if (header) header.style.display = 'none';
    return () => {
      if (header) header.style.display = '';
    };
  }, []);

  useEffect(() => {
    if (navigator.geolocation) {
      navigator.geolocation.getCurrentPosition(
        (position) => {
          setUserLocation({
            lat: position.coords.latitude,
            lng: position.coords.longitude
          });
        },
        (error) => console.error("Geolocation error:", error)
      );
    }
  }, []);

  useEffect(() => {
    if (isDetailModalOpen && selectedVehicle) {
      setLoading(true);
      axios.get(`/tracks/${selectedVehicle.id}`)
        .then(res => {
          setTrackData(res.data);
          setLoading(false);
        })
        .catch(() => {
          setError('Error loading track data');
          setLoading(false);
        });
    }
  }, [isDetailModalOpen, selectedVehicle]);

  const filteredVehicles = vehicles.filter((vehicle) =>
    `${vehicle.id} ${vehicle.number} ${vehicle.driver}`.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const filteredTasks = tasks.filter((task) =>
    `${task.id} ${task.driver} ${task.pickup} ${task.dropoff}`.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const filteredTeams = teams.filter((team) =>
    `${team.id} ${team.name} ${team.manager} ${team.members.join(' ')}`.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleApiLoaded = ({ map, maps }) => {
    mapRef.current = map;
    setMapInstance(map);
    setMapsApi(maps);
  };

  const handleOpenTabs = () => {
    setShowTabs(true);
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Scheduled': return '#FFA500';
      case 'In Progress': return '#3498db';
      case 'Completed': return '#2ecc71';
      case 'LUX': return '#9b59b6';
      case 'In Maintenance': return '#e74c3c';
      case 'Active': return '#4caf50';
      case 'On Break': return '#ff9800';
      default: return '#2ecc71';
    }
  };

  // Function to draw path between pickup and dropoff locations
  const drawPath = (map, maps, pickup, dropoff) => {
    const path = new maps.Polyline({
      path: [
        { lat: pickup.lat, lng: pickup.lng },
        { lat: dropoff.lat, lng: dropoff.lng }
      ],
      geodesic: true,
      strokeColor: '#4285F4',
      strokeOpacity: 1.0,
      strokeWeight: 3
    });
    path.setMap(map);
    return path;
  };

  useEffect(() => {
    if (mapInstance && mapsApi && activeTab === 'tasks' && selectedTask) {
      const path = drawPath(
        mapInstance, 
        mapsApi, 
        selectedTask.pickupLocation, 
        selectedTask.dropoffLocation
      );
      
      return () => {
        path.setMap(null);
      };
    }
  }, [mapInstance, mapsApi, activeTab, selectedTask]);

  return (
    <div style={{ display: 'flex', height: '100vh' }}>
      <div style={{ display: 'none' }}>
        WebSocket status: {isConnected ? 'Connected' : 'Disconnected'}
      </div>

      <style>{`
        .icon-tab {
          position: relative;
          display: flex;
          flex-direction: column;
          align-items: center;
          cursor: pointer;
          padding: 8px;
          border-radius: 4px;
        }
        .icon-tab .hover-label {
          position: absolute;
          top: 100%;
          left: 50%;
          transform: translateX(-50%);
          background: #333;
          color: white;
          font-size: 11px;
          padding: 3px 6px;
          border-radius: 4px;
          opacity: 0;
          pointer-events: none;
          transition: opacity 0.2s;
          white-space: nowrap;
          z-index: 100;
          margin-top: 5px;
        }
        .icon-tab:hover .hover-label {
          opacity: 1;
        }
        .tab-button {
          padding: 8px 16px;
          border: none;
          background: none;
          cursor: pointer;
          font-size: 14px;
          border-bottom: 2px solid transparent;
        }
        .tab-button.active {
          border-bottom: 2px solid #34c759;
          color: #34c759;
          font-weight: 500;
        }
        .status-badge {
          display: inline-block;
          padding: 2px 8px;
          border-radius: 4px;
          font-size: 12px;
          font-weight: 500;
          color: white;
        }
        .list-item {
          padding: 12px 0;
          border-bottom: 1px solid #eee;
          cursor: pointer;
          transition: background-color 0.2s;
        }
        .list-item:hover {
          background-color: #f9f9f9;
        }
        .list-item.selected {
          background-color: #f9f9f9;
        }
        .avatar {
          width: 32px;
          height: 32px;
          border-radius: 50%;
          margin-right: 10px;
          object-fit: cover;
          background-color: #eee;
          display: flex;
          align-items: center;
          justify-content: center;
          color: white;
          font-weight: bold;
          font-size: 14px;
        }
      `}</style>

      {/* Left Sidebar */}
      <div style={{ width: '350px', borderRight: '1px solid #ddd', padding: '1rem', overflowY: 'auto' }}>
        <div style={{ display: 'flex', gap: '0.5rem', alignItems: 'center', marginBottom: '1rem' }}>
          {[
            { icon: 'directions_car', label: 'vehicles' },
            { icon: 'person', label: 'teams' },
            { icon: 'task', label: 'tasks' }
          ].map((item, idx) => (
            <div 
              key={idx}
              className={`icon-tab ${activeTab === item.label ? 'active' : ''}`}
              onClick={() => setActiveTab(item.label)}
              style={{ 
                cursor: 'pointer',
                background: activeTab === item.label ? 'transparent' : 'transparent'
              }}
            >
              <span 
                className="material-symbols-outlined"
                style={{
                  fontSize: 26,
                  color: activeTab === item.label ? '#34c759' : '#666',
                }}
              >
                {item.icon}
              </span>
              <span className="hover-label">
                {item.label.charAt(0).toUpperCase() + item.label.slice(1)}
              </span>
            </div>
          ))}

          <button 
            onClick={handleOpenTabs}
            style={{
              background: 'none',
              border: 'none',
              padding: 0,
              marginLeft: 'auto',
              cursor: 'pointer'
            }}
            title="Open Settings"
          >
            <span className="material-symbols-outlined" style={{ fontSize: 26, color: '#000' }}>
              settings
            </span>
          </button>
        </div>

        <div style={{ position: 'relative', marginBottom: '1.5rem' }}>
          <span className="material-symbols-outlined" style={{
            position: 'absolute', left: '10px', top: '50%', transform: 'translateY(-50%)', color: '#666'
          }}>search</span>
          <input
            type="text"
            placeholder={`Filter by ${activeTab === 'vehicles' ? 'Vehicle ID, Number or Driver' : 
                         activeTab === 'tasks' ? 'Task ID, Driver or Location' : 'Team Name or Member'}`}
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            style={{
              width: '100%', padding: '8px 12px 8px 32px',
              borderRadius: '4px', border: '1px solid #ddd', fontSize: '14px',
              backgroundColor: '#fff'
            }}
          />
        </div>

        {activeTab === 'vehicles' ? (
          filteredVehicles.map((vehicle) => (
            <div
              key={vehicle.id}
              onClick={() => setSelectedVehicle(vehicle)}
              className={`list-item ${selectedVehicle?.id === vehicle.id ? 'selected' : ''}`}
            >
              <div style={{ display: 'flex', alignItems: 'center' }}>
                <div className="avatar">
                  <img 
                    src={getAvatarUrl(vehicle.driver)} 
                    alt={vehicle.driver.charAt(0)} 
                    style={{ width: '100%', height: '100%', borderRadius: '50%' }}
                  />
                </div>
                <div style={{ flex: 1 }}>
                  <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                    <div style={{ fontWeight: 600, fontSize: '15px' }}>{vehicle.driver}</div>
                    <span 
                      className="status-badge" 
                      style={{ backgroundColor: getStatusColor(vehicle.status) }}
                    >
                      {vehicle.status}
                    </span>
                  </div>
                  <div style={{ fontSize: '13px', color: '#333', margin: '4px 0' }}>{vehicle.number}</div>
                  <div style={{ fontSize: '11px', color: '#666' }}>{vehicle.lastUpdated}</div>
                </div>
              </div>
            </div>
          ))
        ) : activeTab === 'tasks' ? (
          filteredTasks.map((task) => (
            <div
              key={task.id}
              onClick={() => setSelectedTask(task)}
              className={`list-item ${selectedTask?.id === task.id ? 'selected' : ''}`}
            >
              <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                <div style={{ fontWeight: 600, fontSize: '15px' }}>{task.driver}</div>
                <span 
                  className="status-badge" 
                  style={{ backgroundColor: getStatusColor(task.status) }}
                >
                  {task.status}
                </span>
              </div>
              <div style={{ fontSize: '13px', color: '#333', margin: '4px 0' }}>
                <div style={{ display: 'flex', alignItems: 'center' }}>
                  <span className="material-symbols-outlined" style={{ fontSize: '16px', marginRight: '6px', color: '#4caf50' }}>location_on</span>
                  <span style={{ overflow: 'hidden', textOverflow: 'ellipsis', whiteSpace: 'nowrap' }}>{task.pickup}</span>
                </div>
                <div style={{ display: 'flex', alignItems: 'center', marginTop: '4px' }}>
                  <span className="material-symbols-outlined" style={{ fontSize: '16px', marginRight: '6px', color: '#f44336' }}>flag</span>
                  <span style={{ overflow: 'hidden', textOverflow: 'ellipsis', whiteSpace: 'nowrap' }}>{task.dropoff}</span>
                </div>
              </div>
              <div style={{ fontSize: '11px', color: '#666' }}>{task.time}</div>
            </div>
          ))
        ) : (
          filteredTeams.map((team) => (
            <div
              key={team.id}
              onClick={() => setSelectedTeam(team)}
              className={`list-item ${selectedTeam?.id === team.id ? 'selected' : ''}`}
            >
              <div style={{ display: 'flex', alignItems: 'center' }}>
                <div className="avatar">
                  <img 
                    src={getAvatarUrl(team.name)} 
                    alt={team.name.charAt(0)} 
                    style={{ width: '100%', height: '100%', borderRadius: '50%' }}
                  />
                </div>
                <div style={{ flex: 1 }}>
                  <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                    <div style={{ fontWeight: 600, fontSize: '15px' }}>{team.name}</div>
                    <span 
                      className="status-badge" 
                      style={{ backgroundColor: getStatusColor(team.status) }}
                    >
                      {team.status}
                    </span>
                  </div>
                  <div style={{ fontSize: '13px', color: '#333', margin: '4px 0' }}>Manager: {team.manager}</div>
                  <div style={{ fontSize: '11px', color: '#666' }}>Members: {team.members.join(', ')}</div>
                </div>
              </div>
            </div>
          ))
        )}
      </div>

      {/* Main Map Area */}
      <div style={{ flex: 1, position: 'relative' }}>
        <div 
          onClick={() => setIsSidebarOpen(!isSidebarOpen)}
          style={{
            position: 'absolute',
            top: '10px',
            right: isSidebarOpen ? '360px' : '10px',
            zIndex: 1000,
            background: 'white',
            borderRadius: '50%',
            width: '40px',
            height: '40px',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            boxShadow: '0 2px 5px rgba(0,0,0,0.2)',
            cursor: 'pointer',
            transition: 'right 0.3s ease'
          }}
        >
          <span className="material-symbols-outlined" style={{ fontSize: '20px',paddingLeft:'10px', color: '#000' }}>
            {isSidebarOpen ? 'close' : 'notifications'}
          </span>
        </div>

        <GoogleMapReact
          bootstrapURLKeys={{ key: 'AIzaSyDe-1PlmNj0J4kGAbZBJcelb_mlUof-6us' }}
          defaultCenter={userLocation}
          defaultZoom={13}
          options={{ fullscreenControl: false }}
          yesIWantToUseGoogleMapApiInternals
          onGoogleApiLoaded={handleApiLoaded}
        >
          <CurrentLocationMarker lat={userLocation.lat} lng={userLocation.lng} />

          {Object.entries(realtimeMarkers).map(([deviceId, marker]) => (
            <CustomMarker key={deviceId} lat={marker.lat} lng={marker.lng}>
              <div style={{ textAlign: 'center', fontSize: '10px', color: 'black' }}>
                <div style={{
                  backgroundColor: '#fff',
                  padding: '2px 6px',
                  borderRadius: '4px',
                  marginTop: '2px',
                  display: 'inline-flex',
                  alignItems: 'center',
                  gap: '4px'
                }}>
                  <span>Speed:</span>
                  <span>{marker.speed ?? 0}</span>
                </div>
                <img
                  src="/car-black.png"
                  alt="car icon"
                  style={{ width: '40px', height: '40px', marginTop: '2px' }}
                />
              </div>
            </CustomMarker>
          ))}

          {activeTab === 'tasks' && selectedTask && (
            <>
              <CustomMarker lat={selectedTask.pickupLocation.lat} lng={selectedTask.pickupLocation.lng}>
                <span className="material-symbols-outlined" style={{ fontSize: '32px', color: '#4caf50' }}>
                  location_on
                </span>
              </CustomMarker>
              <CustomMarker lat={selectedTask.dropoffLocation.lat} lng={selectedTask.dropoffLocation.lng}>
                <span className="material-symbols-outlined" style={{ fontSize: '32px', color: '#f44336' }}>
                  flag
                </span>
              </CustomMarker>
            </>
          )}
        </GoogleMapReact>

        {/* Right Sidebar - Alerts Panel */}
        <div style={{
          width: isSidebarOpen ? '350px' : '0',
          borderLeft: '1px solid #eee',
          background: '#fff',
          overflow: 'hidden',
          transition: 'width 0.3s ease',
          display: 'flex',
          flexDirection: 'column',
          height: '100%',
          position: 'absolute',
          right: 0,
          top: 0
        }}>
          {isSidebarOpen && (
            <div style={{ 
              flex: 1, 
              display: 'flex', 
              flexDirection: 'column',
              padding: '16px',
              overflow: 'hidden'
            }}>
              {/* Search filter */}
              <div style={{ position: 'relative', marginBottom: '1.5rem' }}>
                <span className="material-symbols-outlined" style={{
                  position: 'absolute', left: '10px', top: '50%', transform: 'translateY(-50%)', color: '#666'
                }}>search</span>
                <input
                  type="text"
                  placeholder="Filter alerts"
                  value={rightSearch}
                  onChange={(e) => setRightSearch(e.target.value)}
                  style={{
                    width: '100%', padding: '8px 12px 8px 32px',
                    borderRadius: '4px', border: '1px solid #ddd', fontSize: '14px',
                    backgroundColor: '#fff'
                  }}
                />
              </div>

              {/* Alerts list */}
              <div style={{ 
                flex: 1,
                overflowY: 'auto',
                borderRadius: '8px',
                backgroundColor: '#fff',
              }}>
                {alerts.length > 0 ? (
                  alerts.map((alert, index) => (
                    <div key={index} style={{
                      padding: '12px 16px',
                      borderBottom: '1px solid #eee',
                      transition: 'background-color 0.2s',
                      ':hover': {
                        backgroundColor: '#f1f1f1'
                      }
                    }}>
                      <div style={{ 
                        display: 'flex',
                        justifyContent: 'space-between',
                        marginBottom: '4px'
                      }}>
                        <span style={{ 
                          fontSize: '14px',
                          fontWeight: '500',
                          color: '#333'
                        }}>
                          {alert.split(':')[1]?.trim() || alert}
                        </span>
                        <span style={{ 
                          fontSize: '11px',
                          color: '#999',
                          whiteSpace: 'nowrap',
                          marginLeft: '8px'
                        }}>
                          {alert.split(':')[0]}
                        </span>
                      </div>
                      <div style={{ 
                        fontSize: '12px',
                        color: '#666',
                        lineHeight: '1.4'
                      }}>
                        {alert.includes('has been') ? (
                          <span style={{ color: '#1976d2' }}>{alert}</span>
                        ) : alert.includes('Watch') ? (
                          <span style={{ color: '#d32f2f' }}>{alert}</span>
                        ) : (
                          alert
                        )}
                      </div>
                    </div>
                  ))
                ) : (
                  <div style={{
                    padding: '24px',
                    textAlign: 'center',
                    color: '#999'
                  }}>
                    No alerts to display
                  </div>
                )}
              </div>

              {/* Footer */}
              <div style={{
                marginTop: '12px',
                fontSize: '11px',
                color: '#999',
                textAlign: 'center'
              }}>
                RDEN - Real-time Dispatch Event Notifications
              </div>
            </div>
          )}
        </div>
      </div>

      {isDetailModalOpen && (
        <Modal title="Track Details" onClose={() => setIsDetailModalOpen(false)}>
          {loading ? (
            <div>Loading...</div>
          ) : error ? (
            <div style={{ color: 'red' }}>{error}</div>
          ) : (
            <div>
              <p><strong>Device ID:</strong> {trackData?.device_id || selectedVehicle.id}</p>
              <p><strong>Vehicle:</strong> {trackData?.vehicle_name || selectedVehicle.number}</p>
              <p><strong>Status:</strong> {trackData?.status || 'Active'}</p>
              <p><strong>Last Update:</strong> {trackData?.last_updated ? new Date(trackData.last_updated).toLocaleString() : 'Just now'}</p>
              <p><strong>Location:</strong> {`Lat: ${selectedVehicle.currentLocation.lat}, Lng: ${selectedVehicle.currentLocation.lng}`}</p>
              {trackData?.speed && <p><strong>Speed:</strong> {trackData.speed} km/h</p>}
              {trackData?.battery_level && <p><strong>Battery Level:</strong> {trackData.battery_level}%</p>}
            </div>
          )}
        </Modal>
      )}
    </div>
  );
};

const CustomMarker = ({ children }) => (
  <div style={{
    transform: 'translate(-50%, -50%)',
    fontSize: '20px'
  }}>
    {children}
  </div>
);

const CurrentLocationMarker = () => (
  <div style={{
    width: '12px', height: '12px', background: '#4285F4',
    border: '2px solid white', borderRadius: '50%',
    boxShadow: '0 0 6px rgba(66, 133, 244, 0.6)', transform: 'translate(-50%, -50%)'
  }} />
);

export default TrackView;