import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { storeUserData } from '../Login/AuthStorage';

function GoogleSignIn() {
  const navigate = useNavigate();
  const [errorMessage, setErrorMessage] = useState('');
  const [googleApiLoaded, setGoogleApiLoaded] = useState(false);

  useEffect(() => {
    const script = document.createElement('script');
    script.src = 'https://accounts.google.com/gsi/client';
    script.async = true;
    script.defer = true;
    script.onload = () => setGoogleApiLoaded(true);
    document.body.appendChild(script);
    return () => {
      document.body.removeChild(script);
    };
  }, []);

  useEffect(() => {
    if (googleApiLoaded) {
      window.google.accounts.id.initialize({
        client_id: process.env.REACT_APP_GOOGLE_CLIENT_ID,
        callback: handleCredentialResponse,
        auto_select: false,
      });
      window.google.accounts.id.renderButton(
        document.getElementById('buttonDiv'),
        { theme: 'outline', size: 'large' }
      );
      window.google.accounts.id.prompt();
    }
  }, [googleApiLoaded]);

  const handleCredentialResponse = async (response) => {
    try {
      const backendResponse = await axios.post('/auth/google', { token: response.credential });
      if (backendResponse.status === 200) {
        const userData = backendResponse.data.user;
        storeUserData({ ...userData, token: backendResponse.data.token });
        
        // Redirect based on role and account info
        if (parseInt(userData.roles) === 1) {
          navigate('/saas/dash');
          return;
        }
        if (!userData.account_id) {
          navigate('/setup');
        } else {
          if (userData.account_type >= 1) {
            navigate('/dash');
          } else {
            navigate('/billing');
          }
        }
      } else {
        setErrorMessage('Validation failed. Please try again.');
      }
    } catch (error) {
      console.error("Error during Google sign-in", error.response || error.message);
      setErrorMessage(error.response?.data?.error || "An error occurred during sign-in.");
    }
  };

  return (
    <div>
      {errorMessage && <p style={{ color: 'red' }}>{errorMessage}</p>}
      <div id="buttonDiv" className="g_id_signin"
           data-type="standard"
           data-size="large"
           data-theme="outline"
           data-text="sign_in_with"
           data-shape="rectangular"
           data-logo_alignment="left">
        {/* The Google Sign-In button will be rendered here */}
      </div>
    </div>
  );
}

export default GoogleSignIn;
