

// src/components/Signup/VerifyAndSetPassword.js
import React, { useState, useEffect } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import axios from 'axios';
import queryString from 'query-string';
import { storeUserData } from '../Login/AuthStorage';
import { useDispatch } from 'react-redux';

const VerifyAndSetPassword = () => {
  const [verificationCode, setVerificationCode] = useState('');
  const [password, setPassword] = useState('');
  const [isPasswordVisible, setIsPasswordVisible] = useState(false);
  const [errorMessage, setErrorMessage] = useState('');
  const [successMessage, setSuccessMessage] = useState('');
  const location = useLocation();
  const navigate = useNavigate();
  const [title, setTitle] = useState('');
  const [mode, setMode] = useState('new');
  const [email, setEmail] = useState('');
  const [referralCode, setReferralCode] = useState('');
  const dispatch = useDispatch();

  useEffect(() => {
    // If user already logged in, redirect.
    const userId = localStorage.getItem('user_id');
    if (userId) {
      navigate('/dash');
    }
  }, [navigate]);

  useEffect(() => {
    const queryParams = queryString.parse(location.search);
    const qEmail = queryParams.email || '';
    const code = queryParams.code || '';
    const referral = queryParams.referral || '';
    
    setVerificationCode(code);
    setReferralCode(referral);

    // Set email from either URL params or navigation state
    const emailSource = qEmail || location.state?.email || '';
    if (emailSource) {
      setEmail(emailSource);
    }

    if (location.pathname.includes('/invite')) {
      setTitle('Welcome to thought!');
      setMode('invite');
    } else if (location.pathname.includes('/verify')) {
      setTitle('Signup! Email Verification');
    } else if (location.pathname.includes('/reset-password')) {
      setTitle('Reset Password');
      setMode('reset');
    }
  }, [location]);

  const handleSubmit = async () => {
    if (!isPasswordValid()) {
      setErrorMessage('Password must have at least 8 characters, including at least one number and one special character (@#$%, etc.).');
      return;
    }
    try {
      let response;
      if (mode === 'reset') {
        response = await axios.post('/users/reset-password/', {
          email,
          verificationCode,
          password,
        });
      } else {
        response = await axios.post('/users/', {
          email,
          verificationCode,
          password,
          mode,
          referralCode: mode === 'invite' ? referralCode : undefined,
        });
      }

      if (response.status === 200) {
        setSuccessMessage('Your password has been set successfully.');
        const userData = response.data.user;
        storeUserData({ ...userData, token: response.data.token }, dispatch);
        
        if (userData.account_id === "") {
          navigate('/setup');
        } else {
          navigate('/dash?message=Your account has been successfully created.');
        }
      } else {
        setErrorMessage('Validation failed. Please try again.');
      }
    } catch (error) {
      console.error('Error during verification:', error);
      setErrorMessage(error.response?.data?.message || 'Email (or) Verification code mismatch. Please try again. If you have already completed signup, Signin.');
    }
  };

  const isPasswordValid = () => {
    const minLength = 8;
    const hasNumbers = /\d/.test(password);
    const hasSpecialChar = /[!@#$%^&*(),.?":{}|<>]/.test(password);
    return password.length >= minLength && hasNumbers && hasSpecialChar;
  };

  const togglePasswordVisibility = () => {
    setIsPasswordVisible((prev) => !prev);
  };

  return (
    <div className="container">
      <h2>{title}</h2>
      <p>Enter the verification code sent to your email and set your password to proceed.</p>
      {errorMessage && <p className="error">{errorMessage}</p>}
      {successMessage && <p className="success">{successMessage}</p>}
      <input 
        type="text" 
        placeholder="Your Email" 
        value={email} 
        readOnly 
      />
      <input 
        type="text" 
        placeholder="6-digit code from your email" 
        value={verificationCode} 
        onChange={(e) => setVerificationCode(e.target.value)} 
      />
      <input 
        type={isPasswordVisible ? "text" : "password"} 
        placeholder="Password (minimum 8 characters)" 
        value={password} 
        onChange={(e) => setPassword(e.target.value)} 
      />
      <span onClick={togglePasswordVisibility} className="toggleVisibility">
        {isPasswordVisible ? 'Hide' : 'Show'} password
      </span>
      <br/><br/>
      <button onClick={handleSubmit} className="button">Continue</button>
    </div>
  );
};

export default VerifyAndSetPassword;