// SetupForm.js
import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { withTheme } from '@rjsf/core';
import MaterialUITheme from '@rjsf/material-ui';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';
import useCommonEnums from '../common/CommonData';

const Form = withTheme(MaterialUITheme);

// Custom widget
const IconCheckboxesWidget = (props) => {
  const { id, options = {}, value = [], onChange } = props;
  const { enumOptions = [] } = options;

  const handleClick = (optionValue) => {
    let newValue = Array.isArray(value) ? [...value] : [];
    if (newValue.includes(optionValue)) {
      newValue = newValue.filter(v => v !== optionValue);
    } else {
      newValue.push(optionValue);
    }
    onChange(newValue);
  };

  return (
    <div id={id} style={{ display: 'flex', flexWrap: 'wrap', gap: '10px', marginTop: '10px' }}>
      {enumOptions.map(option => {
        const isSelected = value.includes(option.value);
        return (
          <div
            key={option.value}
            onClick={() => handleClick(option.value)}
            style={{
              border: '1px solid #ccc',
              padding: '10px 15px',
              cursor: 'pointer',
              backgroundColor: isSelected ? '#e0e0e0' : '#fff',
              display: 'flex',
              alignItems: 'center',
              gap: '6px',
              borderRadius: '6px',
              minWidth: '140px',
              marginBottom: '10px'
            }}
          >
            <span className="material-symbols-outlined">chevron_right</span>
            {option.label}
          </div>
        );
      })}
    </div>
  );
};

const SetupForm = () => {
  const navigate = useNavigate();
  const { enums, loading, error } = useCommonEnums();

  const userId = localStorage.getItem('user_id') || '';

  const [formData, setFormData] = useState({
    user_id: userId,
    email: localStorage.getItem('email') || '',
    mobile: localStorage.getItem('mobile') || ''
  });

  const [step, setStep] = useState(1);
  const [errorMessage, setErrorMessage] = useState('');

  // Handle loading and error
  if (loading) return <p>Loading setup form...</p>;
  if (error || !enums) return <p>Error loading setup data. Please refresh.</p>;

  const step1Schema = {
    title: "Account Information",
    type: "object",
    required: ["user_id", "account_name", "title", "industry"],
    properties: {
      user_id: { type: "string", title: "User ID" },
      account_name: { type: "string", title: "Organization Name" },
      title: { type: "string", title: "Your Title" },
      industry: {
        type: "string",
        title: "Industry",
        enum: enums.industries,
        enumNames: enums.industries,
      },
      workforce_count: {
        type: "string",
        title: "Number of Employees",
        enum: enums.workforce_range,
        enumNames: enums.workforce_range
      },
      fleet_count: {
        type: "string",
        title: "Fleet Size",
        enum: enums.workforce_range,
        enumNames: enums.workforce_range
      },
      revenues: {
        type: "string",
        title: "Revenues",
        enum: enums.revenues,
        enumNames: enums.revenues
      },
      address: {
        type: "object",
        title: "Address",
        properties: {
          street: { type: "string", title: "Street" },
          city: { type: "string", title: "City" },
          state: { type: "string", title: "State" },
          country: { type: "string", title: "Country" },
          postal_code: { type: "string", title: "Postal Code" }
        }
      }
    }
  };

  const step1UiSchema = {
    user_id: { "ui:widget": "hidden" },
    industry: { "ui:placeholder": "Select your industry" },
    workforce_count: { "ui:placeholder": "Select number of employees" },
    fleet_count: { "ui:placeholder": "Select fleet size" },
    revenues: { "ui:placeholder": "Select revenue range" }
  };

  const step2Schema = {
    title: "Capabilities",
    type: "object",
    required: ["purposes_of_use", "fleet_capabilities", "workforce_capabilities"],
    properties: {
      purposes_of_use: {
        type: "array",
        title: "Purpose of Use",
        items: {
          type: "string",
          enum: enums.purposes_of_use
        },
        uniqueItems: true
      },
      fleet_capabilities: {
        type: "array",
        title: "Fleet Capabilities",
        items: {
          type: "string",
          enum: enums.fleet_capabilities
        },
        uniqueItems: true
      },
      workforce_capabilities: {
        type: "array",
        title: "Workforce Capabilities",
        items: {
          type: "string",
          enum: enums.workforce_capabilities
        },
        uniqueItems: true
      }
    }
  };

  const step2UiSchema = {
    purposes_of_use: {
      "ui:widget": "IconCheckboxesWidget"
    },
    fleet_capabilities: {
      "ui:widget": "IconCheckboxesWidget"
    },
    workforce_capabilities: {
      "ui:widget": "IconCheckboxesWidget"
    }
  };

  const step3Schema = {
    title: "Personal Information",
    type: "object",
    required: ["name", "email", "mobile"],
    properties: {
      name: { type: "string", title: "Name" },
      email: { type: "string", title: "Email" },
      mobile: { type: "string", title: "Mobile Number" },
      website: { type: "string", title: "Company Domain (Website)" },
      email_domain: { type: "string", title: "Email Domain" }
    }
  };

  const onSubmitStep = async ({ formData: stepData }) => {
    const updatedData = { ...formData, ...stepData };
    setFormData(updatedData);

    if (step < 3) {
      setStep(step + 1);
    } else {
      try {
        const response = await axios.post('/accounts/', updatedData);
        if (response.status === 200) {
          localStorage.setItem('account_id', response.data.account_id || response.data._id);
          navigate(`/dash?setup=true`);
        } else {
          setErrorMessage('Issue occurred. Please try again.');
        }
      } catch (error) {
        console.error('Error setting up account:', error);
        setErrorMessage('Failed to set up account. Please check your input.');
      }
    }
  };

  const onBack = () => {
    if (step > 1) setStep(step - 1);
  };

  const getCurrentSchema = () => {
    if (step === 1) return step1Schema;
    if (step === 2) return step2Schema;
    if (step === 3) return step3Schema;
    return {};
  };

  const getCurrentUiSchema = () => {
    if (step === 1) return step1UiSchema;
    if (step === 2) return step2UiSchema;
    return {};
  };

  return (
    <div style={{ maxWidth: 600, margin: "0 auto" }}>
      <h2>Step {step} of 3</h2>
      {errorMessage && <p className="error" style={{ color: 'red' }}>{errorMessage}</p>}
      <Form
        schema={getCurrentSchema()}
        uiSchema={getCurrentUiSchema()}
        formData={formData}
        onSubmit={onSubmitStep}
        widgets={{ IconCheckboxesWidget }}
        validator={validator}
      >
        <div style={{ display: 'flex', justifyContent: 'space-between' }}>
          {step > 1 && (
            <button type="button" onClick={onBack}>
              Back
            </button>
          )}
          <button type="submit">
            {step === 3 ? 'Submit' : 'Next'}
          </button>
        </div>
      </Form>
    </div>
  );
};

export default SetupForm;


// import React, { useState } from 'react';
// import { useNavigate } from 'react-router-dom';
// import { withTheme } from '@rjsf/core';
// import MaterialUITheme from '@rjsf/material-ui';
// import validator from '@rjsf/validator-ajv8';
// import axios from 'axios';
// import CommonData from '../common/CommonData';

// const Form = withTheme(MaterialUITheme);

// // Custom widget that renders clickable boxes with the Google Material icon.
// const IconCheckboxesWidget = (props) => {
//   const { id, options = {}, value = [], onChange } = props;
//   const { enumOptions = [] } = options;

//   const handleClick = (optionValue) => {
//     let newValue = Array.isArray(value) ? [...value] : [];
//     if (newValue.includes(optionValue)) {
//       newValue = newValue.filter(v => v !== optionValue);
//     } else {
//       newValue.push(optionValue);
//     }
//     onChange(newValue);
//   };

//   return (
//     <div id={id} style={{ display: 'flex', flexWrap: 'wrap', gap: '10px' }}>
//       {enumOptions.map(option => {
//         const isSelected = value.includes(option.value);
//         return (
//           <div 
//             key={option.value}
//             onClick={() => handleClick(option.value)}
//             style={{
//               border: '1px solid #ccc',
//               padding: '10px',
//               cursor: 'pointer',
//               backgroundColor: isSelected ? '#e0e0e0' : '#fff',
//               display: 'flex',
//               alignItems: 'center',
//               gap: '5px'
//             }}
//           >
//             <span className="material-symbols-outlined">chevron_right</span>
//             {option.label}
//           </div>
//         );
//       })}
//     </div>
//   );
// };

// const SetupForm = () => {
//   const navigate = useNavigate();
//   const enums = CommonData();
//   console.log("check this common data",CommonData)
//   console.log("check this enum values",enums)
//   const userId = localStorage.getItem('user_id') || '';

//   // Define the specific enum data as requested
//   const industryOptions = [
//     "Transportation & Mobility",
//     "Logistics & Supply Chain",
//     "Healthcare & Medical Services",
//     "Education & Institutions",
//     "Government & Public Sector",
//     "Telecommunications & Broadband",
//     "Oil, Gas & Energy",
//     "Construction & Infrastructure",
//     "Home & Facility Services",
//     "Field & On-Demand Services",
//     "Retail & Hyperlocal Delivery",
//     "Food & Beverage",
//     "Courier & Express Delivery",
//     "Waste Management & Recycling",
//     "Security & Emergency Response",
//     "Tourism & Hospitality",
//     "Event Management & Entertainment",
//     "Rental & Leasing Services",
//     "Manufacturing & Industrial Operations",
//     "Automotive & Maintenance Services",
//     "Professional & Workforce Services",
//     "Agriculture & Farming",
//     "Real Estate & Property Management",
//     "Banking & Financial Services",
//     "Utilities & Public Works",
//     "Others"
//   ];

// const workforceCountOptions = [
//   "0-10",
//   "11-100", 
//   "101-500",
//   "501-1000",
//   "1001-10000",
//   "10000+"
// ];

//  const fleetSizeOptions = [
//   "0-10",
//   "11-100",
//   "101-500",
//   "501-1000",
//   "1001-10000",
//   "10000+"
// ];
//   const revenueOptions = [
//   "0-1M",
//   "1-10M",
//   "10-100M",
//   "100-500M",
//   "500M-1B",
//   "1B-100B",
//   "100B+"
// ];

//   // Step 1: Account Information Schema (includes user_id as hidden)
//   const step1Schema = {
//     title: "Account Information",
//     type: "object",
//     required: ["user_id", "account_name", "title", "industry"],
//     properties: {
//       user_id: { type: "string", title: "User ID" },
//       account_name: { type: "string", title: "Organization Name" },
//       title: { type: "string", title: "Your Title" },
//       industry: { 
//         type: "string", 
//         title: "Industry",
//         enum: industryOptions,
//         enumNames: industryOptions,
//       },
//       workforce_count: {
//         type: "string",
//         title: "Number of Employees",
//         enum: workforceCountOptions, 
//         enumNames: workforceCountOptions,
//       },
//       fleet_count: {
//         type: "string",
//         title: "Fleet Size",
//         enum: fleetSizeOptions, 
//         enumNames: fleetSizeOptions,
//       },
//       revenues: {
//         type: "string",
//         title: "Revenues",
//         enum: revenueOptions,
//         enumNames: revenueOptions,
//       },
//       address: {
//         type: "object",
//         title: "Address",
//         properties: {
//           street: { type: "string", title: "Street" },
//           city: { type: "string", title: "City" },
//           state: { type: "string", title: "State" },
//           country: { type: "string", title: "Country" },
//           postal_code: { type: "string", title: "Postal Code" }
//         }
//       }
//     }
//   };

//   const step1UiSchema = {
//     user_id: { "ui:widget": "hidden" },
//     industry: { "ui:placeholder": "Select your industry" },
//     workforce_count: { "ui:placeholder": "Select number of employees" },
//     fleet_count: { "ui:placeholder": "Select fleet size" },
//     revenues: { "ui:placeholder": "Select revenue range" }
//   };

//   // Step 2: Capabilities Schema
//   const step2Schema = {
//     title: "Capabilities",
//     type: "object",
//     properties: {
//       purposes_of_use: {
//         type: "array",
//         title: "Purpose of Use",
//         items: { 
//           type: "string", 
//           enum: enums.purposes_of_use 
//         },
//         uniqueItems: true
//       },
//       fleet_capabilities: {
//         type: "array",
//         title: "Fleet Capabilities",
//         items: { 
//           type: "string", 
//           enum: enums.fleet_capabilities 
//         },
//         uniqueItems: true
//       },
//       workforce_capabilities: {
//         type: "array",
//         title: "Workforce Capabilities",
//         items: { 
//           type: "string", 
//           enum: enums.workforce_capabilities 
//         },
//         uniqueItems: true
//       }
//     }
//   };

//   const step2UiSchema = {
//     purposes_of_use: { "ui:widget": "IconCheckboxesWidget" },
//     fleet_capabilities: { "ui:widget": "IconCheckboxesWidget" },
//     workforce_capabilities: { "ui:widget": "IconCheckboxesWidget" }
//   };

//   // Step 3: Personal Information Schema, including Company Domain and Email Domain.
//   const step3Schema = {
//     title: "Personal Information",
//     type: "object",
//     required: ["name", "email", "mobile"],
//     properties: {
//       name: { type: "string", title: "Name" },
//       email: { type: "string", title: "Email" },
//       mobile: { type: "string", title: "Mobile Number" },
//       website: { type: "string", title: "Company Domain (Website)", default: "" },
//       email_domain: { type: "string", title: "Email Domain", default: "" }
//     }
//   };

//   // Prepopulate formData with user_id, email, and mobile from localStorage if available.
//   const [formData, setFormData] = useState({ 
//     user_id: userId, 
//     email: localStorage.getItem('email') || '', 
//     mobile: localStorage.getItem('mobile') || '' 
//   });
  
//   const [step, setStep] = useState(1);
//   const [errorMessage, setErrorMessage] = useState('');

//   const onSubmitStep = async ({ formData: stepData }) => {
//     const updatedData = { ...formData, ...stepData };
//     setFormData(updatedData);
//     if (step < 3) {
//       setStep(step + 1);
//     } else {
//       try {
//         // Final submission using axios.
//         const response = await axios.post('/accounts/', updatedData);
//         if (response.status === 200) {
//           console.log("first check the account values",response.data)
//           localStorage.setItem('account_id', response.data.account_id);
//           //localStorage.setItem('account_id', response.data._id);

//           navigate(`/dash?setup=true`);
//         } else {
//           setErrorMessage('Issue occurred. Please try again.');
//         }
//       } catch (error) {
//         console.error('Error setting up account:', error);
//         setErrorMessage('Failed to set up account. Please check your input.');
//       }
//     }
//   };

//   const onBack = () => {
//     if (step > 1) setStep(step - 1);
//   };

//   const getCurrentSchema = () => {
//     if (step === 1) return step1Schema;
//     if (step === 2) return step2Schema;
//     if (step === 3) return step3Schema;
//   };

//   const getCurrentUiSchema = () => {
//     if (step === 1) return step1UiSchema;
//     if (step === 2) return step2UiSchema;
//     return {};
//   };

//   return (
//     <div style={{ maxWidth: 600, margin: "0 auto" }}>
//       <h2>Step {step} of 3</h2>
//       {errorMessage && <p className="error" style={{ color: 'red' }}>{errorMessage}</p>}
//       <Form
//         schema={getCurrentSchema()}
//         uiSchema={getCurrentUiSchema()}
//         formData={formData}
//         onSubmit={onSubmitStep}
//         widgets={{ IconCheckboxesWidget }}
//         validator={validator}
//       >
//         <div style={{ display: 'flex', justifyContent: 'space-between' }}>
//           {step > 1 && (
//             <button type="button" onClick={onBack}>
//               Back
//             </button>
//           )}
//           <button type="submit">
//             {step === 3 ? 'Submit' : 'Next'}
//           </button>
//         </div>
//       </Form>
//     </div>
//   );
// };

// export default SetupForm;