import React, { useState, useEffect } from 'react';
import axios from 'axios';
import {
  Button,
  Grid,
  Typography,
  FormControlLabel,
  Switch,
  Divider,
} from '@mui/material';

const SystemSettings = () => {
  const [settings, setSettings] = useState({
    save_logs: true,
    analytics_tracking_id_web: '',
    analytics_tracking_id_app: '',
    chat_support_enabled: false,
  });

  const id = '67fec97f95809e100c9954d4';
  const apiBase = `monitoring/${id}`;

  useEffect(() => {
    const fetchSettings = async () => {
      try {
        const response = await axios.get(apiBase);
        const data = response.data;
        setSettings({
          save_logs: data.save_logs !== undefined ? Boolean(data.save_logs) : true,
          analytics_tracking_id_web: data.analytics_tracking_id_web || '',
          analytics_tracking_id_app: data.analytics_tracking_id_app || '',
          chat_support_enabled: data.chat_support_enabled !== undefined ? Boolean(data.chat_support_enabled) : false,
        });
      } catch (error) {
        console.error('Failed to fetch system settings:', error);
      }
    };

    fetchSettings();
  }, [apiBase]);

  const handleChange = (event) => {
    const { name, value, type, checked } = event.target;
    setSettings({
      ...settings,
      [name]: type === 'checkbox' ? checked : value,
    });
  };

  const handleSubmit = async () => {
    try {
      const response = await axios.post(apiBase, settings);
      console.log('Settings saved:', response.data);
    } catch (error) {
      console.error('Failed to save system settings:', error);
    }
  };

  return (
    <div className="settings-container">
      <Typography variant="subtitle1" gutterBottom className="settings-section-title">
        <span className="material-symbols-outlined settings-icon">storage</span>
        System Monitoring
      </Typography>

      <Grid container spacing={2} sx={{ mb: 3 }}>
        <Grid item xs={12}>
          <FormControlLabel
            control={
<Switch
                checked={settings.save_logs}
                onChange={handleChange}
                name="save_logs"
                // ✅ Changed switch color to green
                sx={{
                  '& .MuiSwitch-switchBase.Mui-checked': {
                    color: '#4CAF50',
                  },
                  '& .MuiSwitch-switchBase.Mui-checked + .MuiSwitch-track': {
                    backgroundColor: '#4CAF50',
                  },
                }}
              />
            }
            label={<span className="settings-label">Enable System Logs</span>}
          />
        </Grid>
      </Grid>

      <Divider className="settings-divider" />

      <Typography variant="subtitle1" gutterBottom className="settings-section-title">
        <span className="material-symbols-outlined settings-icon">analytics</span>
        Analytics & Tracking
      </Typography>

      <Grid container spacing={2} sx={{ mb: 3 }}>
        <Grid item xs={12} sm={6}>
          <label className="settings-label">Web Tracking ID</label>
          <input
            className="settings-input"
            type="text"
            name="analytics_tracking_id_web"
            value={settings.analytics_tracking_id_web}
            onChange={handleChange}
          />
        </Grid>
        <Grid item xs={12} sm={6}>
          <label className="settings-label">App Tracking ID</label>
          <input
            className="settings-input"
            type="text"
            name="analytics_tracking_id_app"
            value={settings.analytics_tracking_id_app}
            onChange={handleChange}
          />
        </Grid>
      </Grid>

      <Divider className="settings-divider" />

      <Typography variant="subtitle1" gutterBottom className="settings-section-title">
        <span className="material-symbols-outlined settings-icon">support_agent</span>
        Support Widget
      </Typography>

      <Grid container spacing={2} sx={{ mb: 3 }}>
        <Grid item xs={12}>
          <FormControlLabel
            control={
 <Switch
                checked={settings.chat_support_enabled}
                onChange={handleChange}
                name="chat_support_enabled"
                // ✅ Changed switch color to green
                sx={{
                  '& .MuiSwitch-switchBase.Mui-checked': {
                    color: '#4CAF50',
                  },
                  '& .MuiSwitch-switchBase.Mui-checked + .MuiSwitch-track': {
                    backgroundColor: '#4CAF50',
                  },
                }}
              />
            }
            label={<span className="settings-label">Enable Chat/Support Widgets</span>}
          />
        </Grid>
      </Grid>

<Button
        variant="contained"
        onClick={handleSubmit}
        // ✅ Changed button text to lowercase
        sx={{
          backgroundColor: '#4CAF50', // ✅ Green button
          '&:hover': {
            backgroundColor: '#45A049'
          },
          color: '#fff',
          textTransform: 'none' // ✅ Keeps text in lowercase
        }}
      >
        save changes
      </Button>
    </div>
  );
};

export default SystemSettings;