import React, { useState, useEffect } from 'react';

const SubscriptionSettings = ({ data, onSave }) => {
  const [planOverview, setPlanOverview] = useState({
    plan_name: '',
    plan_id: '',
    plan_description: '',
    start_date: '',
    end_date: '',
    is_active: true
  });

  const [usageLimits, setUsageLimits] = useState({
    max_users: 0,
    max_devices: 0,
    max_geofences: 0,
    max_tasks_per_day: 0
  });

  const [renewalPreferences, setRenewalPreferences] = useState({
    auto_renewal: true,
    renewal_cycle: 'yearly',
    reminder_days: 15
  });

  const [invoiceInfo, setInvoiceInfo] = useState({
    last_invoice_id: '',
    last_invoice_date: '',
    next_invoice_date: '',
    download_invoice_url: ''
  });

  const [payment, setPayment] = useState({
    payment_mode: '',
    stripe_customer_id: ''
  });
  const [saveSuccess, setSaveSuccess] = useState(false);

  useEffect(() => {
    if (data.subscription) {
      setPlanOverview(data.subscription.plan_overview || {
        plan_name: '',
        plan_id: '',
        plan_description: '',
        start_date: '',
        end_date: '',
        is_active: true
      });

      setUsageLimits(data.subscription.usage_limits || {
        max_users: 0,
        max_devices: 0,
        max_geofences: 0,
        max_tasks_per_day: 0
      });

      setRenewalPreferences(data.subscription.renewal_preferences || {
        auto_renewal: true,
        renewal_cycle: 'yearly',
        reminder_days: 15
      });

      setInvoiceInfo(data.subscription.invoice_info || {
        last_invoice_id: '',
        last_invoice_date: '',
        next_invoice_date: '',
        download_invoice_url: ''
      });

      setPayment(data.subscription.payment || {
        payment_mode: '',
        stripe_customer_id: ''
      });
    }
  }, [data]);

  const handleSubmit = () => {
    onSave('subscription', {
      plan_overview: planOverview,
      usage_limits: usageLimits,
      renewal_preferences: renewalPreferences,
      invoice_info: invoiceInfo,
      payment: payment
    });
    setSaveSuccess(true);
    setTimeout(() => setSaveSuccess(false), 3000);
  };

  const handlePlanOverviewChange = (field, value) => {
    setPlanOverview(prev => ({ ...prev, [field]: value }));
  };

  const handleUsageLimitsChange = (field, value) => {
    setUsageLimits(prev => ({ ...prev, [field]: parseInt(value) || 0 }));
  };

  const handleRenewalPreferencesChange = (field, value) => {
    setRenewalPreferences(prev => ({ ...prev, [field]: value }));
  };

  const handlePaymentChange = (field, value) => {
    setPayment(prev => ({ ...prev, [field]: value }));
  };

  return (
    <div className="settings-container">
      {/* Plan Overview Section */}
      <div className="settings-section">
        <h3 className="settings-section-title">
          <span className="material-symbols-outlined settings-icon">description</span>
          Plan Overview
        </h3>
        
        <div className="settings-row">
          <div className="settings-input-group">
            <label className="settings-label">Plan Name</label>
            <input
              type="text"
              className="settings-input"
              value={planOverview.plan_name}
              onChange={(e) => handlePlanOverviewChange('plan_name', e.target.value)}
            />
          </div>
          <div className="settings-input-group">
            <label className="settings-label">Plan ID</label>
            <input
              type="text"
              className="settings-input"
              value={planOverview.plan_id}
              disabled
            />
          </div>
        </div>

        <div style={{ marginBottom: '15px' }}>
          <label className="settings-label">Plan Description</label>
          <textarea
            className="settings-input settings-textarea"
            value={planOverview.plan_description}
            onChange={(e) => handlePlanOverviewChange('plan_description', e.target.value)}
          />
        </div>

        <div className="settings-row">
          <div className="settings-input-group">
            <label className="settings-label">Start Date</label>
            <input
              type="date"
              className="settings-input"
              value={planOverview.start_date}
              onChange={(e) => handlePlanOverviewChange('start_date', e.target.value)}
            />
          </div>
          <div className="settings-input-group">
            <label className="settings-label">End Date</label>
            <input
              type="date"
              className="settings-input"
              value={planOverview.end_date}
              onChange={(e) => handlePlanOverviewChange('end_date', e.target.value)}
            />
          </div>
        </div>

        <div style={{ display: 'flex', alignItems: 'center', gap: '10px' }}>
          <input
            type="checkbox"
            id="is_active"
            checked={planOverview.is_active}
            onChange={(e) => handlePlanOverviewChange('is_active', e.target.checked)}
            style={{ width: 'auto' }}
          />
          <label htmlFor="is_active" className="settings-label">Active Plan</label>
        </div>
      </div>

      {/* Usage Limits Section */}
      <div className="settings-section">
        <h3 className="settings-section-title">
          <span className="material-symbols-outlined settings-icon">people</span>
          Usage Limits
        </h3>
        
        <div className="settings-row">
          <div className="settings-input-group">
            <label className="settings-label">Max Users</label>
            <input
              type="number"
              min="0"
              className="settings-input"
              value={usageLimits.max_users}
              onChange={(e) => handleUsageLimitsChange('max_users', e.target.value)}
            />
          </div>
          <div className="settings-input-group">
            <label className="settings-label">Max Devices</label>
            <input
              type="number"
              min="0"
              className="settings-input"
              value={usageLimits.max_devices}
              onChange={(e) => handleUsageLimitsChange('max_devices', e.target.value)}
            />
          </div>
        </div>

        <div className="settings-row">
          <div className="settings-input-group">
            <label className="settings-label">Max Geofences</label>
            <input
              type="number"
              min="0"
              className="settings-input"
              value={usageLimits.max_geofences}
              onChange={(e) => handleUsageLimitsChange('max_geofences', e.target.value)}
            />
          </div>
          <div className="settings-input-group">
            <label className="settings-label">Max Tasks/Day</label>
            <input
              type="number"
              min="0"
              className="settings-input"
              value={usageLimits.max_tasks_per_day}
              onChange={(e) => handleUsageLimitsChange('max_tasks_per_day', e.target.value)}
            />
          </div>
        </div>
      </div>

      {/* Renewal Preferences Section */}
      <div className="settings-section">
        <h3 className="settings-section-title">
          <span className="material-symbols-outlined settings-icon">event</span>
          Renewal Preferences
        </h3>
        
        <div className="settings-row">
          <div className="settings-input-group">
            <div style={{ display: 'flex', alignItems: 'center', gap: '10px' }}>
              <input
                type="checkbox"
                id="auto_renewal"
                checked={renewalPreferences.auto_renewal}
                onChange={(e) => handleRenewalPreferencesChange('auto_renewal', e.target.checked)}
                style={{ width: 'auto' }}
              />
              <label htmlFor="auto_renewal" className="settings-label">Auto Renewal</label>
            </div>
          </div>
          <div className="settings-input-group">
            <label className="settings-label">Renewal Cycle</label>
            <select
              className="settings-select"
              value={renewalPreferences.renewal_cycle}
              onChange={(e) => handleRenewalPreferencesChange('renewal_cycle', e.target.value)}
            >
              <option value="monthly">Monthly</option>
              <option value="quarterly">Quarterly</option>
              <option value="yearly">Yearly</option>
            </select>
          </div>
        </div>

        <div className="settings-row">
          <div className="settings-input-group">
            <label className="settings-label">Reminder Days Before Renewal</label>
            <input
              type="number"
              min="1"
              max="30"
              className="settings-input"
              value={renewalPreferences.reminder_days}
              onChange={(e) => handleRenewalPreferencesChange('reminder_days', e.target.value)}
            />
          </div>
          <div className="settings-input-group"></div>
        </div>
      </div>

      {/* Payment Section */}
      <div className="settings-section">
        <h3 className="settings-section-title">
          <span className="material-symbols-outlined settings-icon">credit_card</span>
          Payment Information
        </h3>
        
        <div className="settings-row">
          <div className="settings-input-group">
            <label className="settings-label">Payment Mode</label>
            <select
              className="settings-select"
              value={payment.payment_mode}
              onChange={(e) => handlePaymentChange('payment_mode', e.target.value)}
            >
              <option value="">Select payment mode</option>
              <option value="stripe">Stripe</option>
              <option value="bank">Bank Transfer</option>
              <option value="paypal">PayPal</option>
            </select>
          </div>
          <div className="settings-input-group">
            <label className="settings-label">Stripe Customer ID</label>
            <input
              type="text"
              className="settings-input"
              value={payment.stripe_customer_id}
              disabled
            />
          </div>
        </div>
      </div>

      {/* Invoice Information Section */}
      <div className="settings-section">
        <h3 className="settings-section-title">
          <span className="material-symbols-outlined settings-icon">receipt</span>
          Invoice Information
        </h3>
        
        <div className="settings-row">
          <div className="settings-input-group">
            <label className="settings-label">Last Invoice ID</label>
            <input
              type="text"
              className="settings-input"
              value={invoiceInfo.last_invoice_id}
              disabled
            />
          </div>
          <div className="settings-input-group">
            <label className="settings-label">Last Invoice Date</label>
            <input
              type="date"
              className="settings-input"
              value={invoiceInfo.last_invoice_date}
              disabled
            />
          </div>
        </div>

        <div className="settings-row">
          <div className="settings-input-group">
            <label className="settings-label">Next Invoice Date</label>
            <input
              type="date"
              className="settings-input"
              value={invoiceInfo.next_invoice_date}
              disabled
            />
          </div>
          <div className="settings-input-group">
            {invoiceInfo.download_invoice_url && (
              <button
                className="settings-button"
                onClick={() => window.open(invoiceInfo.download_invoice_url, '_blank')}
              >
                Download Invoice
              </button>
            )}
          </div>
        </div>
      </div>

      <div style={{ display: 'flex', alignItems: 'center', gap: '15px' }}>
        <button
          type="button"
          className="settings-button"
          onClick={handleSubmit}
        >
          Save Changes
        </button>

        {saveSuccess && (
          <div className="settings-success-message">
            <span className="material-symbols-outlined settings-success-icon">
              check_circle
            </span>
            Saved successfully!
          </div>
        )}
      </div>
    </div>
  );
};

export default SubscriptionSettings;