// SettingsPage.js
import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Tabs from '@mui/material/Tabs';
import Tab from '@mui/material/Tab';
import Typography from '@mui/material/Typography';
import Divider from '@mui/material/Divider';
import Paper from '@mui/material/Paper';

import AccountSettings from './AccountSettings';
import SubscriptionSettings from './SubscriptionSettings';
import AvailabilitySettings from './AvailabilitySettings';
import CommunicationSettings from './CommunicationSettings';
import IntegrationSettings from './IntegrationSettings';
import AppLinksSettings from './AppLinksSettings';
import LanguageSettings from './LanguageSettings';
import DisplaySettings from './DisplaySettings';
import SystemSettings from './SystemSettings';


const SettingsPage = () => {
  const [activeTab, setActiveTab] = useState(0);
  const [loading, setLoading] = useState({});
  const [settingsData, setSettingsData] = useState({});

  // Map tabs to their API endpoints
  const tabEndpoints = {
    0: 'account_settings',
    1: 'subscription',
    2: 'availability',
    3: 'communication',
    4: 'integrations',
    5: 'applinks',
    6: 'language_regional',
    7: 'ui',
    8: 'monitoring'
  };

  const fetchTabData = async (tabIndex) => {
    const endpoint = tabEndpoints[tabIndex];
    if (!endpoint) return;

    setLoading(prev => ({ ...prev, [endpoint]: true }));
    
    try {
      const accountId = localStorage.getItem('account_id');
      const response = await axios.get(`${endpoint}/${accountId}`);
      setSettingsData(prev => ({ 
        ...prev, 
        [endpoint]: response.data 
      }));
    } catch (error) {
      console.error(`Error fetching ${endpoint} settings:`, error);
    } finally {
      setLoading(prev => ({ ...prev, [endpoint]: false }));
    }
  };

  useEffect(() => {
    // Load initial tab data when component mounts
    fetchTabData(activeTab);
  }, []);

  const handleTabChange = (event, newValue) => {
    setActiveTab(newValue);
    fetchTabData(newValue);
  };

  const handleSaveSettings = async (section, data) => {
    try {
      const accountId = localStorage.getItem('account_id');
      await axios.post(`${section}/${accountId}`, data);
      setSettingsData(prev => ({ ...prev, [section]: data }));
      alert(`${section.replace('_', ' ')} settings updated successfully!`);
    } catch (error) {
      console.error(`Error updating ${section} settings:`, error);
      alert(`Failed to update ${section} settings`);
    }
  };

  const tabs = [
    { label: 'Account', icon: 'manage_accounts', component: AccountSettings },
    { label: 'Subscription', icon: 'paid', component: SubscriptionSettings },
    { label: 'Availability & Tracking', icon: 'location_on', component: AvailabilitySettings },
    { label: 'Communication', icon: 'notifications', component: CommunicationSettings },
    { label: 'Integrations', icon: 'extension', component: IntegrationSettings },
    { label: 'App Links', icon: 'link', component: AppLinksSettings },
    { label: 'Language & Regional', icon: 'language', component: LanguageSettings },
    { label: 'Display & UI', icon: 'palette', component: DisplaySettings },
    { label: 'System & Monitoring', icon: 'monitoring', component: SystemSettings },
  ];

  const currentTabEndpoint = tabEndpoints[activeTab];
  const isTabLoading = loading[currentTabEndpoint];

  return (
    <div style={{ display: 'flex', minHeight: '100vh', overflowY: 'auto', marginTop: '16px' }}>
      <Paper
        elevation={2}
        style={{
          width: '250px',
          minWidth: '250px',
          maxWidth: '250px',
          marginRight: '16px',
          backgroundColor: '#ffffff',
          flexShrink: 0,
        }}
      >
        <Tabs
          orientation="vertical"
          variant="scrollable"
          value={activeTab}
          onChange={handleTabChange}
          aria-label="Settings vertical tabs"
          TabIndicatorProps={{ style: { backgroundColor: '#34c759', width: 4, left: 0 } }}
          style={{ height: '100%' }}
        >
          {tabs.map((tab, index) => {
            const isActive = activeTab === index;
            const tabEndpoint = tabEndpoints[index];
            const tabIsLoading = loading[tabEndpoint];
            
            return (
              <Tab
                key={index}
                label={
                  <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                    <span
                      className="material-symbols-outlined"
                      style={{ color: isActive ? 'green' : 'black' }}
                    >
                      {tab.icon}
                    </span>
                    <Typography
                      variant="body2"
                      style={{ color: isActive ? 'green' : 'black' }}
                    >
                      {tab.label}
                      {tabIsLoading && ' (Loading...)'}
                    </Typography>
                  </div>
                }
                style={{
                  alignItems: 'flex-start',
                  justifyContent: 'flex-start',
                  textTransform: 'none',
                  backgroundColor: isActive ? '#ffffff' : 'transparent',
                  padding: '8px 16px',
                  width: '100%',
                  maxWidth: '100%',
                  minHeight: '48px',
                }}
              />
            );
          })}
        </Tabs>
      </Paper>

      <div style={{ flexGrow: 1, padding: '16px' }}>
        <Typography variant="h6" gutterBottom>
          {tabs[activeTab].label}
          {isTabLoading && ' (Loading...)'}
        </Typography>
        <Divider style={{ marginBottom: '16px' }} />
        {isTabLoading ? (
          <div>Loading {tabs[activeTab].label.toLowerCase()} settings...</div>
        ) : (
          React.createElement(tabs[activeTab].component, {
            data: settingsData,
            onSave: handleSaveSettings,
          })
        )}
      </div>
    </div>
  );
};

export default SettingsPage;