import React, { useEffect, useState } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';

const schema = {
  type: 'object',
  required: ['default_language', 'time_zone', 'date_format', 'number_format'],
  properties: {
    default_language: {
      type: 'string',
      title: 'Default Platform Language',
      enum: ['English', 'Spanish', 'French', 'German', 'Arabic']
    },
    time_zone: {
      type: 'string',
      title: 'Time Zone',
      enum: ['Asia/Kolkata', 'UTC', 'EST', 'PST', 'CET', 'GMT']
    },
    date_format: {
      type: 'string',
      title: 'Date Format',
      enum: ['MM/DD/YYYY', 'DD/MM/YYYY', 'DD-MM-YYYY', 'YYYY-MM-DD']
    },
    number_format: {
      type: 'string',
      title: 'Number Format',
      enum: ['1,234.56', '1.234,56', '1 234,56']
    }
  }
};

const uiSchema = {
  default_language: { 'ui:placeholder': 'Select default language' },
  time_zone: { 'ui:placeholder': 'Select time zone' },
  date_format: { 'ui:placeholder': 'Choose date format' },
  number_format: { 'ui:placeholder': 'Choose number format' }
};

const transformErrors = () => [];

const LanguageSettings = ({ userId = '67fec97f95809e100c9954d4', onSave }) => {
  const [formData, setFormData] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchLanguageSettings = async () => {
      try {
        const res = await fetch(`language_regional/${userId}`);
        const data = await res.json();
        setFormData(data);
      } catch (error) {
        console.error('Failed to load language settings', error);
      } finally {
        setLoading(false);
      }
    };

    fetchLanguageSettings();
  }, [userId]);

  const handleSubmit = async ({ formData }) => {
    try {
      await fetch(`language_regional/${userId}`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(formData)
      });
      onSave?.('language_regional', formData);
    } catch (error) {
      console.error('Failed to update language settings', error);
    }
  };

  if (loading || !formData) {
    return <div className="settings-container">Loading...</div>;
  }

  return (
    <div className="settings-container">
      <h3 className="settings-section-title">
        <span className="material-symbols-outlined settings-icon">
          language
        </span>
        Language & Regional Settings
      </h3>

      <Form
        schema={schema}
        uiSchema={uiSchema}
        formData={formData}
        onChange={({ formData }) => setFormData(formData)}
        onSubmit={handleSubmit}
        validator={validator}
        transformErrors={transformErrors}
        showErrorList={false}
      >
        <button
          type="submit"
          className="settings-button"
        >
          Save Changes
        </button>
      </Form>
    </div>
  );
};

export default LanguageSettings;