import React, { useState, useEffect } from 'react';
import { Button, Grid, Typography, FormControlLabel, Divider } from '@mui/material';
import { styled } from '@mui/material/styles';
import Switch from '@mui/material/Switch';

// ✅ Custom green switch
const GreenSwitch = styled(Switch)(({ theme }) => ({
  '& .MuiSwitch-switchBase.Mui-checked': {
    color: '#4CAF50',
    '&:hover': {
      backgroundColor: '#4CAF5033',
    },
  },
  '& .MuiSwitch-switchBase.Mui-checked + .MuiSwitch-track': {
    backgroundColor: '#4CAF50',
  },
}));

const IntegrationSettings = ({ data, onSave }) => {
  const [integrations, setIntegrations] = useState({
    sms_service_enabled: false,
    email_service_enabled: false,
    firebase_enabled: false,
    google_maps_enabled: false,
    payment_gateway_enabled: false
  });

  useEffect(() => {
    if (data.integrations) {
      setIntegrations({
        sms_service_enabled: data.integrations.sms_service_enabled ?? false,
        email_service_enabled: data.integrations.email_service_enabled ?? false,
        firebase_enabled: data.integrations.firebase_enabled ?? false,
        google_maps_enabled: data.integrations.google_maps_enabled ?? false,
        payment_gateway_enabled: data.integrations.payment_gateway_enabled ?? false
      });
    }
  }, [data]);

  const handleChange = (event) => {
    setIntegrations({
      ...integrations,
      [event.target.name]: event.target.checked,
    });
  };

  const handleSubmit = () => {
    onSave('integrations', integrations);
  };

  return (
    <div className="settings-container">
      <Typography variant="h6" gutterBottom className="settings-section-title">
        <span className="material-symbols-outlined settings-icon">cloud</span>
        Service Integrations
      </Typography>

      <Typography variant="body2" color="text.secondary" sx={{ mb: 3, fontSize: '0.85rem' }}>
        Toggle the active services configured in the Apps section.
      </Typography>

      <Grid container spacing={2} sx={{ mb: 3 }}>
        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={integrations.sms_service_enabled}
                onChange={handleChange}
                name="sms_service_enabled"
              />
            }
            label={
              <span className="settings-label">
                <span className="material-symbols-outlined settings-icon">sms</span>
                SMS Service (Twilio)
              </span>
            }
          />
        </Grid>

        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={integrations.email_service_enabled}
                onChange={handleChange}
                name="email_service_enabled"
              />
            }
            label={
              <span className="settings-label">
                <span className="material-symbols-outlined settings-icon">email</span>
                Email Service
              </span>
            }
          />
        </Grid>

        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={integrations.firebase_enabled}
                onChange={handleChange}
                name="firebase_enabled"
              />
            }
            label={
              <span className="settings-label">
                <span className="material-symbols-outlined settings-icon">sms</span>
                Firebase (Push & Logs)
              </span>
            }
          />
        </Grid>

        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={integrations.google_maps_enabled}
                onChange={handleChange}
                name="google_maps_enabled"
              />
            }
            label={
              <span className="settings-label">
                <span className="material-symbols-outlined settings-icon">map</span>
                Google Maps (Map & Geolocation)
              </span>
            }
          />
        </Grid>

        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={integrations.payment_gateway_enabled}
                onChange={handleChange}
                name="payment_gateway_enabled"
              />
            }
            label={
              <span className="settings-label">
                <span className="material-symbols-outlined settings-icon">payments</span>
                Payment Gateway
              </span>
            }
          />
        </Grid>
      </Grid>

      <Button
        variant="contained"
        onClick={handleSubmit}
        sx={{
          backgroundColor: '#4CAF50',
          '&:hover': {
            backgroundColor: '#45A049'
          },
          color: '#fff',
          textTransform: 'none'
        }}
      >
        Save Changes
      </Button>
    </div>
  );
};

export default IntegrationSettings;
