import React, { useState, useEffect } from 'react';
import {
  Button,
  Grid,
  Typography,
  FormControlLabel,
  Divider
} from '@mui/material';
import { styled } from '@mui/material/styles';
import Switch from '@mui/material/Switch';

// ✅ Custom Green Switch
const GreenSwitch = styled(Switch)(({ theme }) => ({
  '& .MuiSwitch-switchBase.Mui-checked': {
    color: '#4CAF50',
    '&:hover': {
      backgroundColor: '#4CAF5033',
    },
  },
  '& .MuiSwitch-switchBase.Mui-checked + .MuiSwitch-track': {
    backgroundColor: '#4CAF50',
  },
}));

const CommunicationSettings = ({ data, onSave }) => {
  const [notifications, setNotifications] = useState({
    sms_enabled: false,
    email_enabled: false,
    show_otp: false,
    skip_customer_email: false,
    skip_workforce_email: false
  });

  useEffect(() => {
    if (data.communication) {
      setNotifications({
        sms_enabled: data.communication.sms_enabled ?? false,
        email_enabled: data.communication.email_enabled ?? false,
        show_otp: data.communication.show_otp ?? false,
        skip_customer_email: data.communication.skip_customer_email ?? false,
        skip_workforce_email: data.communication.skip_workforce_email ?? false,
      });
    }
  }, [data]);

  const handleChange = (event) => {
    const { name, checked } = event.target;
    setNotifications(prev => ({
      ...prev,
      [name]: checked,
    }));
  };

  const handleSubmit = () => {
    onSave('communication', notifications);
  };

  return (
    <div className="settings-container">
      <Typography variant="h6" gutterBottom className="settings-section-title">
        <span className="material-symbols-outlined settings-icon">
          notifications_active
        </span>
        Notification Settings
      </Typography>

      <Grid container spacing={2} sx={{ mb: 3 }}>
        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={notifications.sms_enabled}
                onChange={handleChange}
                name="sms_enabled"
              />
            }
            label={<span className="settings-label">Enable SMS Notifications</span>}
          />
        </Grid>
        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={notifications.email_enabled}
                onChange={handleChange}
                name="email_enabled"
              />
            }
            label={<span className="settings-label">Enable Email Notifications</span>}
          />
        </Grid>
        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={notifications.show_otp}
                onChange={handleChange}
                name="show_otp"
              />
            }
            label={<span className="settings-label">Enable OTP Display</span>}
          />
        </Grid>
      </Grid>

      <Divider className="settings-divider" />

      <Typography variant="h6" gutterBottom className="settings-section-title">
        <span className="material-symbols-outlined settings-icon">
          email
        </span>
        Email Preferences
      </Typography>

      <Grid container spacing={2} sx={{ mb: 3 }}>
        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={notifications.skip_customer_email}
                onChange={handleChange}
                name="skip_customer_email"
              />
            }
            label={<span className="settings-label">Skip Emails to Customers</span>}
          />
        </Grid>
        <Grid item xs={12}>
          <FormControlLabel
            control={
              <GreenSwitch
                checked={notifications.skip_workforce_email}
                onChange={handleChange}
                name="skip_workforce_email"
              />
            }
            label={<span className="settings-label">Skip Emails to Workforce</span>}
          />
        </Grid>
      </Grid>

      <Button
        variant="contained"
        onClick={handleSubmit}
        sx={{
          backgroundColor: '#4CAF50',
          '&:hover': {
            backgroundColor: '#45A049',
          },
          color: '#fff',
          textTransform: 'none',
        }}
      >
        Save Changes
      </Button>
    </div>
  );
};

export default CommunicationSettings;
