import React, { useState, useEffect } from 'react';
import axios from 'axios';

// ✅ Added styles object here
const styles = {
  input: {
    padding: '8px',
    fontSize: '14px',
    width: '100%',
    borderRadius: '4px',
    outline: 'none',
    marginBottom: '5px',
  },
  inputGroup: {
    display: 'flex',
    flexDirection: 'column',
    flex: 1,
    marginRight: '15px',
  },
};

const AccountSettings = ({ id = "67fec97f95809e100c9954d4" }) => {
  const [businessProfile, setBusinessProfile] = useState({});
  const [personalInfo, setPersonalInfo] = useState({});
  const [billingDetails, setBillingDetails] = useState({});
  const [errors, setErrors] = useState({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [saveSuccess, setSaveSuccess] = useState(false);

  useEffect(() => {
    axios.get(`account_settings/${id}`)
      .then((res) => {
        const data = res.data;
        setBusinessProfile(data.business_profile || {});
        setPersonalInfo(data.personal_information || {});
        setBillingDetails(data.billing_details || {});
      })
      .catch((err) => {
        console.error("Error fetching account settings:", err);
      });
  }, [id]);

  const validateFields = () => {
    const newErrors = {};

    if (!businessProfile.business_name?.trim()) newErrors.business_name = "Business name is required";
    if (!businessProfile.business_email?.trim()) newErrors.business_email = "Business email is required";
    else if (!/^\S+@\S+\.\S+$/.test(businessProfile.business_email)) newErrors.business_email = "Invalid email format";

    if (!personalInfo.first_name?.trim()) newErrors.first_name = "First name is required";
    if (!personalInfo.email?.trim()) newErrors.email = "Email is required";
    else if (!/^\S+@\S+\.\S+$/.test(personalInfo.email)) newErrors.email = "Invalid email format";

    if (!billingDetails.billing_email?.trim()) newErrors.billing_email = "Billing email is required";
    else if (!/^\S+@\S+\.\S+$/.test(billingDetails.billing_email)) newErrors.billing_email = "Invalid email format";

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setSaveSuccess(false);
    if (!validateFields()) return;

    setIsSubmitting(true);
    try {
      await axios.post(`account_settings/${id}`, {
        business_profile: businessProfile,
        personal_information: personalInfo,
        billing_details: billingDetails,
      });
      setSaveSuccess(true);
      setTimeout(() => setSaveSuccess(false), 3000);
    } catch (error) {
      console.error("Save failed:", error);
    } finally {
      setIsSubmitting(false);
    }
  };

  const inputStyle = (fieldName) => ({
    ...styles.input,
    border: errors[fieldName] ? '1px solid #ff4444' : '1px solid #ddd',
  });

  return (
    <div className="settings-container">
      <form onSubmit={handleSubmit}>
        {/* --- Business Profile Section --- */}
        <div className="settings-section">
          <h3 className="settings-section-title">
            <span className="material-symbols-outlined settings-icon">business_center</span>
            Business Profile
          </h3>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">Business Name*</label>
              <input
                type="text"
                className={`settings-input ${errors.business_name ? 'settings-input-error' : ''}`}
                value={businessProfile.business_name || ''}
                onChange={(e) => setBusinessProfile({ ...businessProfile, business_name: e.target.value })}
              />
              {errors.business_name && <div className="settings-error-message">{errors.business_name}</div>}
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Business Email*</label>
              <input
                type="email"
                className={`settings-input ${errors.business_email ? 'settings-input-error' : ''}`}
                value={businessProfile.business_email || ''}
                onChange={(e) => setBusinessProfile({ ...businessProfile, business_email: e.target.value })}
              />
              {errors.business_email && <div className="settings-error-message">{errors.business_email}</div>}
            </div>
          </div>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">Business Phone</label>
              <input
                type="text"
                className="settings-input"
                value={businessProfile.business_phone || ''}
                onChange={(e) => setBusinessProfile({ ...businessProfile, business_phone: e.target.value })}
              />
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Business Type</label>
              <input
                type="text"
                className="settings-input"
                value={businessProfile.business_type || ''}
                onChange={(e) => setBusinessProfile({ ...businessProfile, business_type: e.target.value })}
              />
            </div>
          </div>

          <div style={{ marginBottom: '15px' }}>
            <label className="settings-label">Business Address</label>
            <input
              type="text"
              className="settings-input"
              style={{ width: '100%' }}
              value={businessProfile.business_address || ''}
              onChange={(e) => setBusinessProfile({ ...businessProfile, business_address: e.target.value })}
            />
          </div>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">Business Registration</label>
              <input
                type="text"
                className="settings-input"
                value={businessProfile.business_registration || ''}
                onChange={(e) => setBusinessProfile({ ...businessProfile, business_registration: e.target.value })}
              />
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Tax ID</label>
              <input
                type="text"
                className="settings-input"
                value={businessProfile.tax_id || ''}
                onChange={(e) => setBusinessProfile({ ...businessProfile, tax_id: e.target.value })}
              />
            </div>
          </div>

          <div>
            <label className="settings-label">Website URL</label>
            <input
              type="url"
              className="settings-input"
              value={businessProfile.website_url || ''}
              onChange={(e) => setBusinessProfile({ ...businessProfile, website_url: e.target.value })}
            />
          </div>
        </div>

        {/* --- Personal Information Section --- */}
        <div className="settings-section">
          <h3 className="settings-section-title">
            <span className="material-symbols-outlined settings-icon">person</span>
            Personal Information
          </h3>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">First Name*</label>
              <input
                type="text"
                className={`settings-input ${errors.first_name ? 'settings-input-error' : ''}`}
                value={personalInfo.first_name || ''}
                onChange={(e) => setPersonalInfo({ ...personalInfo, first_name: e.target.value })}
              />
              {errors.first_name && <div className="settings-error-message">{errors.first_name}</div>}
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Last Name</label>
              <input
                type="text"
                className="settings-input"
                value={personalInfo.last_name || ''}
                onChange={(e) => setPersonalInfo({ ...personalInfo, last_name: e.target.value })}
              />
            </div>
          </div>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">Email*</label>
              <input
                type="email"
                className={`settings-input ${errors.email ? 'settings-input-error' : ''}`}
                value={personalInfo.email || ''}
                onChange={(e) => setPersonalInfo({ ...personalInfo, email: e.target.value })}
              />
              {errors.email && <div className="settings-error-message">{errors.email}</div>}
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Phone Number</label>
              <input
                type="text"
                className="settings-input"
                value={personalInfo.phone_number || ''}
                onChange={(e) => setPersonalInfo({ ...personalInfo, phone_number: e.target.value })}
              />
            </div>
          </div>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">Role</label>
              <input
                type="text"
                className="settings-input"
                value={personalInfo.role || ''}
                onChange={(e) => setPersonalInfo({ ...personalInfo, role: e.target.value })}
              />
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Profile Picture URL</label>
              <input
                type="url"
                className="settings-input"
                value={personalInfo.profile_picture_url || ''}
                onChange={(e) => setPersonalInfo({ ...personalInfo, profile_picture_url: e.target.value })}
              />
            </div>
          </div>
        </div>

        {/* --- Billing Details Section --- */}
        <div className="settings-section">
          <h3 className="settings-section-title">
            <span className="material-symbols-outlined settings-icon">payments</span>
            Billing Details
          </h3>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">Billing Name</label>
              <input
                type="text"
                className="settings-input"
                value={billingDetails.billing_name || ''}
                onChange={(e) => setBillingDetails({ ...billingDetails, billing_name: e.target.value })}
              />
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Billing Email*</label>
              <input
                type="email"
                className={`settings-input ${errors.billing_email ? 'settings-input-error' : ''}`}
                value={billingDetails.billing_email || ''}
                onChange={(e) => setBillingDetails({ ...billingDetails, billing_email: e.target.value })}
              />
              {errors.billing_email && <div className="settings-error-message">{errors.billing_email}</div>}
            </div>
          </div>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">Billing Phone</label>
              <input
                type="text"
                className="settings-input"
                value={billingDetails.billing_phone || ''}
                onChange={(e) => setBillingDetails({ ...billingDetails, billing_phone: e.target.value })}
              />
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Payment Method</label>
              <input
                type="text"
                className="settings-input"
                value={billingDetails.payment_method || ''}
                onChange={(e) => setBillingDetails({ ...billingDetails, payment_method: e.target.value })}
              />
            </div>
          </div>

          <div style={{ marginBottom: '15px' }}>
            <label className="settings-label">Billing Address</label>
            <textarea
              rows="3"
              className="settings-input settings-textarea"
              value={billingDetails.billing_address || ''}
              onChange={(e) => setBillingDetails({ ...billingDetails, billing_address: e.target.value })}
            />
          </div>

          <div className="settings-row">
            <div className="settings-input-group">
              <label className="settings-label">Card Number</label>
              <input
                type="text"
                className="settings-input"
                value={billingDetails.card_number || ''}
                onChange={(e) => setBillingDetails({ ...billingDetails, card_number: e.target.value })}
              />
            </div>
            <div className="settings-input-group">
              <label className="settings-label">Card Expiry</label>
              <input
                type="text"
                className="settings-input"
                value={billingDetails.card_expiry || ''}
                onChange={(e) => setBillingDetails({ ...billingDetails, card_expiry: e.target.value })}
              />
            </div>
            <div style={{ ...styles.inputGroup, flex: '0.5' }}>
              <label className="settings-label">Card CVV</label>
              <input
                type="text"
                className="settings-input"
                value={billingDetails.card_cvv || ''}
                onChange={(e) => setBillingDetails({ ...billingDetails, card_cvv: e.target.value })}
              />
            </div>
          </div>
        </div>

        <div style={{ display: 'flex', alignItems: 'center', gap: '15px' }}>
         <button
  type="submit"
  style={{
    backgroundColor: '#4CAF50', // Green color
    color: 'white',
    padding: '10px 20px',
    border: 'none',
    borderRadius: '4px',
    cursor: 'pointer',
    fontSize: '16px',
    transition: 'none', // Remove any transitions
    // Add other styles you want to maintain
  }}
  disabled={isSubmitting}
>
  save changes
</button>

          {saveSuccess && (
            <div className="settings-success-message">
              <span className="material-symbols-outlined settings-success-icon">
                check_circle
              </span>
              Saved successfully!
            </div>
          )}
        </div>
      </form>
    </div>
  );
};

export default AccountSettings;
