import React, { useState, useEffect } from 'react';
import axios from 'axios';
import ServicesForm from './ServicesForm';
import Modal from '../../Layouts/Modal';

const ServicesUpdate = ({ userId, onClose, onUpdateSuccess }) => {  
  const [serviceData, setServiceData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    axios.get(`/services/${userId}`)
      .then(res => {
        setServiceData(res.data);
        setLoading(false);
      })
      .catch(err => {
        setError("Error fetching service data.");
        setLoading(false);
      });
  }, [userId]);

  const handleSubmit = async (formData) => {
    try {
      const formDataToSend = new FormData();
      
      // Append all non-file fields
      Object.keys(formData).forEach(key => {
        if (key !== 'image' && key !== 'map_icon') {
          formDataToSend.append(key, formData[key]);
        }
      });

      // Handle image update if it's a new file
      if (formData.image && formData.image.startsWith('data:')) {
        const imageFile = dataURLtoFile(formData.image, 'service-image');
        formDataToSend.append('image', imageFile);
      }

      // Handle map icon update if it's a new file
      if (formData.map_icon && formData.map_icon.startsWith('data:')) {
        const iconFile = dataURLtoFile(formData.map_icon, 'map-icon');
        formDataToSend.append('map_icon', iconFile);
      }

      const response = await axios.put(`/services/${userId}`, formDataToSend, {
        headers: {
          'Content-Type': 'multipart/form-data'
        }
      });
      
      alert("Service updated successfully!");
      if (onUpdateSuccess) onUpdateSuccess(response.data);
      onClose();
    } catch (err) {
      console.error("Error updating service:", err);
      alert("Error updating service.");
    }
  };

  // Helper function to convert data URL to File object
  const dataURLtoFile = (dataurl, filename) => {
    const arr = dataurl.split(',');
    const mime = arr[0].match(/:(.*?);/)[1];
    const bstr = atob(arr[1]);
    let n = bstr.length;
    const u8arr = new Uint8Array(n);
    while (n--) {
      u8arr[n] = bstr.charCodeAt(n);
    }
    return new File([u8arr], filename, { type: mime });
  };

  return (
    <Modal title="Edit Service" onClose={onClose}>
      {loading ? (
        <div>Loading service data...</div>
      ) : error ? (
        <div style={{ color: 'red' }}>{error}</div>
      ) : (
        <ServicesForm initialData={serviceData} onSubmit={handleSubmit} />
      )}
    </Modal>
  );
};

export default ServicesUpdate;