import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import UserView from './ServicesView';
import UserUpdate from './ServicesUpdate';
import UserDB from './ServicesDB';
import CommonData from '../../common/CommonData';

const ROLES_HEADER_TO_KEY_MAP = {
  "Service Name": {
    key: "service_title",
    label: "Service Name",
    sortable: true,
    clickable: true // ✅ Make only this field clickable
  },
  "Service Description": { key: "service_description", label: "Service Description", sortable: false },
  "Created Date": { key: "created_date", label: "Created Date", sortable: false },
};

const UsersList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery
  });

  const [selectedUser, setSelectedUser] = useState(null);
  const enums = CommonData();

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    { name: 'created_date_from', type: 'date', placeholder: 'From Date' },
    { name: 'created_date_to', type: 'date', placeholder: 'To Date' },
    {
      name: 'has_cost',
      type: 'dropdown',
      placeholder: 'Has Cost',
      options: [
        { label: 'All', value: '' },
        { label: 'Yes', value: '1' },
        { label: 'No', value: '0' }
      ]
    },
    {
      name: 'is_vehicle_focused',
      type: 'dropdown',
      placeholder: 'Vehicle Focused',
      options: [
        { label: 'All', value: '' },
        { label: 'Yes', value: '1' },
        { label: 'No', value: '0' }
      ]
    }
  ];

  const actionFormats = [
    { label: "view", modal: true, url: "/services/view/{id}" },
    { label: "edit", modal: true, url: "/services/update/{id}" },
    { label: "delete", modal: true, url: "/services/delete/{id}" }
  ];

  const handleActionClick = (userId, action) => {
    if (action.label === "view") {
      setSelectedUser({ userId, modalType: "view" });
    } else if (action.label === "edit") {
      setSelectedUser({ userId, modalType: "update" });
    } else if (action.label === "delete") {
      setSelectedUser({ userId, modalType: "delete" });
    }
  };

  // ✅ Handle cell click on "Service Name"
  const handleCellClick = (header, item) => {
    if (header === "Service Name") {
      const userId = item.user_id || item.id;
      if (userId) {
        console.log("✅ Service Name clicked → opening modal for ID:", userId);
        setSelectedUser({ userId, modalType: "view" });
      } else {
        console.warn("⚠️ Missing user_id in item:", item);
      }
    }
  };

  const handleSearch = (params) => {
    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      created_date_from: params.created_date_from || '',
      created_date_to: params.created_date_to || '',
      has_cost: params.has_cost || '',
      is_vehicle_focused: params.is_vehicle_focused || '',
      skip: 0
    }));
  };

  const handlePageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage
    }));
  };

  const buildFetchUrl = () => {
    let url = `/services/list/all/?skip=${searchParams.skip}&limit=${searchParams.limit}`;
    if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
    if (searchParams.created_date_from) url += `&from=${searchParams.created_date_from}`;
    if (searchParams.created_date_to) url += `&to=${searchParams.created_date_to}`;
    if (searchParams.has_cost) url += `&has_cost=${searchParams.has_cost}`;
    if (searchParams.is_vehicle_focused) url += `&is_vehicle_focused=${searchParams.is_vehicle_focused}`;
    return url;
  };

  const closeModal = () => setSelectedUser(null);

  return (
    <div>
      <SearchPage
        fetchDataUrl={buildFetchUrl()}
        ListComponent={List}
        itemKeyMapping={ROLES_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="user_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick} 
        searchFields={searchFields}
        onSearch={handleSearch}
        onPageChange={handlePageChange}
      />

      {selectedUser?.modalType === "view" && (
        <UserView 
          userId={selectedUser.userId}
          onClose={closeModal}
        />
      )}
      {selectedUser?.modalType === "update" && (
        <UserUpdate 
          userId={selectedUser.userId}
          onClose={closeModal}
          onUpdateSuccess={() => console.log("User updated")}
        />
      )}
      {selectedUser?.modalType === "delete" && (
        <UserDB 
          userId={selectedUser.userId}
          onClose={closeModal}
          onDeleteSuccess={() => console.log("User deleted")}
        />
      )}
    </div>
  );
};

export default UsersList;
