import React, { useState, useEffect, createContext, useContext } from "react";
import axios from 'axios';
import Form from "@rjsf/core";
import validator from "@rjsf/validator-ajv8";

// Create context to manage dropdown state
const DropdownContext = createContext();

// JSON schema for Pricing Rules Form
export const userSchema = {
  title: "Service Form",
  description: "Create or update Service.",
  type: "object",
  required: [
    "service_id",
    "service_title",
    "service_description",
    "capacity_jobs",
    "has_cost",
    "is_vehicle_focused",
  ],
  properties: {
    service_id: {
      type: "string",
      title: "Service Category",
      oneOf: [
        {
          const: "",
          title: "Select Category",
        },
      ],
    },
    service_title: {
      type: "string",
      title: "Service Name",
    },
    service_description: {
      type: "string",
      title: "Service Description",
    },
    skills: {
      type: "array",
      title: "Skills",
      items: {
        type: "string",
        oneOf: [] // Will be populated dynamically
      },
      uniqueItems: true,
    },
    capacity_jobs: {
      type: "number",
      title: "Capacity",
    },
    has_cost: {
      type: "number",
      title: "Has Cost",
      enum: [1, 0],
      enumNames: ["Yes", "No"],
    },
    is_vehicle_focused: {
      type: "number",
      title: "Is Vehicle Focused",
      enum: [1, 0],
      enumNames: ["Yes", "No"],
    },    
    checklist: {
      type: "array",
      title: "Checklist",
      items: {
        type: "string",
        oneOf: [] // Will be populated dynamically
      },
      uniqueItems: true,
    },
    addons: {
      type: "array",
      title: "Addons",
      items: {
        type: "string",
        oneOf: [] // Will be populated dynamically
      },
      uniqueItems: true,
    },
    image: {
      type: "string",
      title: "Service Image",
      format: "data-url",
    },
    map_icon: {
      type: "string",
      title: "Map Icon",
      format: "data-url",
    },
    account_id: {
      type: "string",
      title: "User ID",
      default: "{id}",
    },
  },
};

// Custom dropdown widget with improved dropdown management
const CustomSelectWidget = (props) => {
  const { id, label, value, required, onChange, options, schema } = props;
  const [searchTerm, setSearchTerm] = useState("");
  const [items, setItems] = useState([]);
  const [loading, setLoading] = useState(false);
 
  // Use dropdown context to manage open state
  const { openDropdown, setOpenDropdown } = useContext(DropdownContext);
  const isOpen = openDropdown === id;

  const fetchItems = async () => {
    try {
      setLoading(true);
      let endpoint = '';
     
      if (schema.title === 'Skills') {
        endpoint = 'crud/skills?skip=0&limit=10&q=&status=';
      } else if (schema.title === 'Checklist') {
        endpoint = 'crud/checklist?skip=0&limit=10&q=&status=';
      } else if (schema.title === 'Addons') {
        endpoint = 'crud/addons?skip=0&limit=10&q=&status=';
      }

      const response = await axios.get(endpoint);
      const fetchedItems = response.data?.items || [];
     
      // Ensure we're returning proper string values
      setItems(fetchedItems.map(item => ({
        id: String(item.id || item.skills_id || item.checklist_id || item.addons_id),
        name: String(item.name || '')
      })));
    } catch (error) {
      console.error(`Error fetching ${schema.title}:`, error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (isOpen) {
      fetchItems();
    }
  }, [isOpen]);

  const filteredItems = items.filter(item =>
    item.name.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleSelect = (item) => {
    const currentValue = Array.isArray(value) ? value : [];
    const newValue = currentValue.includes(item.id)
      ? currentValue.filter(v => v !== item.id)
      : [...currentValue, item.id];
   
    onChange(newValue);
  };

  const selectedItems = Array.isArray(value)
    ? items.filter(item => value.includes(item.id))
    : [];

  const toggleDropdown = () => {
    if (isOpen) {
      setOpenDropdown(null);
    } else {
      setOpenDropdown(id);
    }
  };

  return (
    <div className="form-group" style={{ marginBottom: '20px', position: 'relative' }}>
      <div
        style={{
          border: '1px solid #ced4da',
          borderRadius: '4px',
          padding: '8px 12px',
          minHeight: '23px',
          cursor: 'pointer',
          backgroundColor: '#fff',
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'space-between',
          position: 'relative'
        }}
        onClick={toggleDropdown}
      >
        <div style={{
          display: 'flex',
          flexWrap: 'wrap',
          gap: '4px',
          flex: 1,
          overflow: 'hidden'
        }}>
          {selectedItems.length > 0 ? (
            selectedItems.map(item => (
              <span
                key={item.id}
                style={{
                  backgroundColor: '#e9ecef',
                  borderRadius: '4px',
                  padding: '2px 6px',
                  fontSize: '14px',
                  display: 'inline-flex',
                  alignItems: 'center'
                }}
              >
                {item.name}
                <span
                  style={{
                    marginLeft: '4px',
                    cursor: 'pointer',
                    fontSize: '12px'
                  }}
                  onClick={(e) => {
                    e.stopPropagation();
                    handleSelect(item);
                  }}
                >
                  ×
                </span>
              </span>
            ))
          ) : (
            <span style={{ color: '#6c757d' }}>
              Select {label.toLowerCase()}
            </span>
          )}
        </div>
        <span style={{
          color: 'black',
          marginLeft: '8px',
          fontWeight: 'bold',
          fontSize: '12px',
          display: 'inline-block',
          transform: 'scaleX(1.3)'
        }}>
          {isOpen ? '˄' : '˅'}
        </span>
      </div>

      {isOpen && (
        <div
          style={{
            position: 'absolute',
            zIndex: 1000,
            width: '100%',
            maxHeight: '300px',
            overflowY: 'auto',
            backgroundColor: 'white',
            border: '1px solid #ced4da',
            borderRadius: '4px',
            boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
            marginTop: '4px'
          }}
        >
          <div style={{ padding: '8px'}}></div>

          {loading && (
            <div style={{ padding: '8px', textAlign: 'center', color: '#6c757d' }}>
              Loading...
            </div>
          )}

          {!loading && filteredItems.length === 0 && (
            <div style={{ padding: '8px', textAlign: 'center', color: '#6c757d' }}>
              No {label.toLowerCase()} found
            </div>
          )}

          {!loading && filteredItems.length > 0 && (
            <ul style={{ listStyle: 'none', padding: 0, margin: 0 }}>
              {filteredItems.map(item => (
                <li
                  key={item.id}
                  style={{
                    padding: '8px 12px',
                    cursor: 'pointer',
                    backgroundColor: value?.includes(item.id) ? '#f8f9fa' : 'white',
                    borderBottom: '1px solid #f8f9fa',
                    display: 'flex',
                    alignItems: 'center'
                  }}
                  onClick={(e) => {
                    e.stopPropagation();
                    handleSelect(item);
                  }}
                >
                  <input
                    type="radio"
                    checked={value?.includes(item.id)}
                    readOnly
                    style={{
                      marginRight: '8px',
                      accentColor: '#0d6efd'
                    }}
                  />
                  {item.name}
                </li>
              ))}
            </ul>
          )}
        </div>
      )}
    </div>
  );
};


const RadioWidget = ({ id, schema, options, value, required, disabled, readonly, onChange }) => {
  const { enumOptions, enumNames } = options;


  return (
    <div className="form-group" id={id}>
      <label>{schema.title}{required ? '*' : ''}</label>
      <div style={{ display: 'flex', gap: '20px', marginTop: '8px' }}>
        {enumOptions.map((option, i) => (
          <label key={option.value} style={{ display: 'flex', alignItems: 'center' }}>
            <input
              type="radio"
              checked={option.value === value}
              onChange={() => onChange(option.value)}
              disabled={disabled || readonly}
              style={{ marginRight: '5px' }}
            />
            {enumNames[i]}
          </label>
        ))}
      </div>
    </div>
  );
};


const CustomFileWidget = (props) => {
  const { value, onChange } = props;


  const handleChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (event) => {
        onChange(event.target.result);
      };
      reader.readAsDataURL(file);
    }
  };


  return (
    <div>
      {value && (
        <div>
          <img
            src={value}
            alt="Preview"
            style={{ maxWidth: '100px', marginBottom: '10px' }}
          />
          <p>Current: {value.startsWith('data:') ? 'New Upload' : value}</p>
        </div>
      )}
      <input
        type="file"
        onChange={handleChange}
        accept="image/*"
      />
    </div>
  );
};


const DynamicFieldManager = ({ fields, setFields }) => {
  const addField = () => {
    setFields([...fields, { label: '', type: 'text', options: '' }]);
  };


  const removeField = (index) => {
    const newFields = fields.filter((_, i) => i !== index);
    setFields(newFields);
  };


  const updateField = (index, key, value) => {
    const newFields = [...fields];
    newFields[index][key] = value;
    setFields(newFields);
  };


  const canAddNew = fields.every(f => f.label.trim() && f.type &&
    (['select', 'checkbox'].includes(f.type) ? f.options.trim() : true));


  return (
    <div className="dynamic-fields">
      {fields.map((field, index) => (
        <div key={index} className="field-row" style={{ marginBottom: '1rem' }}>
          <input
            type="text"
            placeholder="Field Label"
            value={field.label}
            onChange={(e) => updateField(index, 'label', e.target.value)}
            style={{ marginRight: '1rem', width: '150px' }}
          />
         
          <select
            value={field.type}
            onChange={(e) => updateField(index, 'type', e.target.value)}
            style={{ marginRight: '1rem', width: '120px' }}
          >
            <option value="text">Text</option>
            <option value="textarea">Textarea</option>
            <option value="select">Select</option>
            <option value="checkbox">Checkbox</option>
          </select>


          {['select', 'checkbox'].includes(field.type) && (
            <input
              type="text"
              placeholder="Options (comma separated)"
              value={field.options}
              onChange={(e) => updateField(index, 'options', e.target.value)}
              style={{ marginRight: '1rem', width: '200px' }}
            />
          )}


          <button
            type="button"
            onClick={() => removeField(index)}
            style={{ marginLeft: '0.5rem' }}
          >
            -
          </button>
        </div>
      ))}


      <button
        type="button"
        onClick={addField}
        disabled={!canAddNew}
        style={{ marginTop: '1rem' }}
      >
        Add Field +
      </button>
    </div>
  );
};


export const userUiSchema = {
  image: {
    "ui:widget": "file",
  },
  map_icon: {
    "ui:widget": "file",
  },
  is_vehicle_focused: {
    "ui:widget": "radio",
  },
  has_cost: {
    "ui:widget": "radio",
  },
  user_id: {
    "ui:widget": "hidden",
  },
  account_id: {
    "ui:widget": "hidden",
  },
  checklist: {
    "ui:widget": CustomSelectWidget,
    "ui:options": {
      multiple: true,
    },
  },
  skills: {
    "ui:widget": CustomSelectWidget,
    "ui:options": {
      multiple: true,
    },
  },
  addons: {
    "ui:widget": CustomSelectWidget,
    "ui:options": {
      multiple: true,
    },
  },
  "ui:submitButtonOptions": {
    norender: true,
  },
};


const UserForm = ({ initialData, onSubmit }) => {
  const [formData, setFormData] = useState(initialData || {});
  const [schema, setSchema] = useState(userSchema);
  const [dynamicFields, setDynamicFields] = useState([]);
  const [statusMessage, setStatusMessage] = useState({ text: '', type: '' });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const formRef = React.useRef();
 
  // State to track which dropdown is currently open
  const [openDropdown, setOpenDropdown] = useState(null);


  useEffect(() => {
    if (initialData) {
      const convertedData = {
        ...initialData,
        has_cost: initialData.has_cost ? 1 : 0,
        is_vehicle_focused: initialData.is_vehicle_focused ? 1 : 0
      };
      setFormData(convertedData);
     
      if (initialData.category_inputs) {
        const fields = initialData.category_inputs.map(input => ({
          label: input.label,
          type: input.input_type === 'text' ? 'text' : input.input_type,
          options: ''
        }));
        setDynamicFields(fields);
      }
    }
  }, [initialData]);


  const checkForDuplicateService = async (serviceTitle) => {
    try {
      const response = await axios.get(
        `/services/list/all/?q=${encodeURIComponent(serviceTitle)}`
      );
     
      return response.data.users?.some(service => {
        const isSameTitle = service.service_title.toLowerCase() === serviceTitle.toLowerCase();
        const isDifferentService = !initialData || service._id?.$oid !== initialData._id?.$oid;
        return isSameTitle && isDifferentService;
      });
    } catch (error) {
      console.error('Error checking for duplicate services:', error);
      return false;
    }
  };


  useEffect(() => {
    const fetchChecklist = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getchecklist/${userId}`);


        if (response.data?.users) {
          const checkList = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.name,
          }));


          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              checklist: {
                ...prev.properties.checklist,
                items: {
                  ...prev.properties.checklist.items,
                  oneOf: checkList,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching checklists:", error);
      }
    };


    fetchChecklist();
  }, []);


  useEffect(() => {
    const fetchSkills = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getskills/${userId}`);


        if (response.data?.users) {
          const skills = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.name,
          }));


          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              skills: {
                ...prev.properties.skills,
                items: {
                  ...prev.properties.skills.items,
                  oneOf: skills,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching skills:", error);
      }
    };


    fetchSkills();
  }, []);


  useEffect(() => {
    const fetchAddons = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getaddonlist/${userId}`);


        if (response.data?.users) {
          const addons = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.name,
          }));


          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              addons: {
                ...prev.properties.addons,
                items: {
                  ...prev.properties.addons.items,
                  oneOf: addons,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching addons:", error);
      }
    };


    fetchAddons();
  }, []);


  useEffect(() => {
    const fetchServices = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getservicecategorylist/${userId}`);


        if (response.data?.users) {
          const serviceOptions = [
            ...userSchema.properties.service_id.oneOf,
            ...response.data.users.map((service) => ({
              const: service.user_id,
              title: service.service_category,
            }))
          ];


          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              service_id: {
                ...prev.properties.service_id,
                oneOf: serviceOptions,
              }
            }
          }));
        }
      } catch (error) {
        console.error("Error fetching services:", error);
      }
    };


    fetchServices();
  }, []);


  const handleSubmit = async ({ formData }) => {
    setIsSubmitting(true);
    setStatusMessage({ text: '', type: '' });


    try {
      if (!formData.service_title || formData.service_title.trim().length < 3) {
        throw new Error('Service title must be at least 3 characters long');
      }


      const isDuplicate = await checkForDuplicateService(formData.service_title);
      if (isDuplicate) {
        throw new Error('A service with this name already exists');
      }


      const submissionData = {
        ...formData,
        has_cost: formData.has_cost === 1,
        is_vehicle_focused: formData.is_vehicle_focused === 1,
        dynamicFields: JSON.stringify(
        dynamicFields.map(f => ({
          ...f,
          options: f.options ? f.options.split(',').map(o => o.trim()) : []
        }))
      )
    };


      await onSubmit(submissionData);
     
    } catch (error) {
      console.error('Error submitting form:', error);
      setStatusMessage({
        text: error.message || 'Error submitting form. Please try again.',
        type: 'error'
      });
    } finally {
      setIsSubmitting(false);
    }
  };


  return (
    <DropdownContext.Provider value={{ openDropdown, setOpenDropdown }}>
      <div style={{ maxWidth: "600px", padding: '20px' }}>
        {statusMessage.text && (
          <div
            style={{
              padding: '10px 15px',
              marginBottom: '20px',
              borderRadius: '4px',
              backgroundColor: statusMessage.type === 'error' ? '#ffebee' : '#e8f5e9',
              color: statusMessage.type === 'error' ? '#c62828' : '#2e7d32',
              border: `1px solid ${statusMessage.type === 'error' ? '#ef9a9a' : '#a5d6a7'}`,
              fontWeight: 'bold'
            }}
          >
            {statusMessage.text}
          </div>
        )}


        <Form
          ref={formRef}
          schema={schema}
          uiSchema={userUiSchema}
          formData={formData}
          onChange={({ formData }) => setFormData(formData)}
          onSubmit={handleSubmit}
          validator={validator}
          widgets={{ FileWidget: CustomFileWidget }}
        />
       
        <h3>Additional Fields</h3>
        <DynamicFieldManager
          fields={dynamicFields}
          setFields={setDynamicFields}
        />


        <div style={{ marginTop: '20px', textAlign: 'left' }}>
          <button
            type="button"
            onClick={() => formRef.current.submit()}
            disabled={isSubmitting}
            style={{
              padding: '8px 20px',
              backgroundColor: isSubmitting ? '#e0e0e0' : '#f0f0f0',
              color: isSubmitting ? '#9e9e9e' : '#00c853',
              borderRadius: '4px',
              cursor: isSubmitting ? 'not-allowed' : 'pointer',
              transition: 'all 0.3s ease',
            }}
            onMouseEnter={(e) => {
              if (!isSubmitting) {
                e.currentTarget.style.backgroundColor = '#00c853';
                e.currentTarget.style.color = 'white';
              }
            }}
            onMouseLeave={(e) => {
              if (!isSubmitting) {
                e.currentTarget.style.backgroundColor = '#f0f0f0';
                e.currentTarget.style.color = '#00c853';
              }
            }}
          >
            {isSubmitting ? 'Processing...' : 'Submit'}
          </button>
        </div>
      </div>
    </DropdownContext.Provider>
  );
};


export default UserForm;