import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import CommonTaskList from '../../common/CommonTaskList';


const TAB_CONFIG = [
  { id: 'details', label: 'Dashboard', icon: 'dashboard' },
  { id: 'tasks', label: 'Tasks', icon: 'assignment' },
  { id: 'workforce', label: 'Workforce', icon: 'engineering' }
];


const ScheduleView = ({ scheduleId, onClose }) => {
  const [scheduleData, setScheduleData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('details');
  const [hoveredTab, setHoveredTab] = useState(null);


  useEffect(() => {
    axios.get(`/schedules/${scheduleId}`)
      .then(res => {
        setScheduleData(res.data);
        setLoading(false);
      })
      .catch(() => {
        setError('Error fetching schedule details.');
        setLoading(false);
      });
  }, [scheduleId]);


  const formatDateTime = (dateTime) => {
    if (!dateTime) return 'Not set';
    return new Date(dateTime).toLocaleString();
  };


  const calculateDuration = (start, end) => {
    if (!start || !end) return 'N/A';
    const startTime = new Date(start);
    const endTime = new Date(end);
    const diffMs = endTime - startTime;
    const hours = Math.floor(diffMs / (1000 * 60 * 60));
    const minutes = Math.floor((diffMs % (1000 * 60 * 60)) / (1000 * 60));
    return `${hours}h ${minutes}m`;
  };
 


  const renderTabContent = () => {
    if (!scheduleData) return null;


    switch (activeTab) {
      case 'details':
        return (
           <div className="schedule-details">
            <div style={{ marginTop: '15px' }}>
              <h4>
                <span className="material-symbols-outlined" style={{
                  verticalAlign: 'middle',
                  marginRight: '8px',
                  color: '#333' // Black color for header icon
                }}>
                  dashboard
                </span>
                Schedule Information
              </h4>
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <p><strong>Name:</strong> {scheduleData.schedule_name}</p>
                <p><strong>Description:</strong> {scheduleData.description || 'None'}</p>
                <p><strong>Created:</strong> {formatDateTime(scheduleData.created_date)}</p>
              </div>
            </div>


            <div style={{ marginTop: '15px' }}>
              <h4 style={{
                display: 'flex',
                alignItems: 'center',
                gap: '8px',
                fontWeight: 'bold',
                fontSize: '14px'
              }}>
                <span className="material-symbols-outlined" style={{
                  fontSize: '20px',
                  color: '#333' // Black color for header icon
                }}>
                  schedule
                </span>
                Time Information
              </h4>


              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px',
              }}>
                <p><strong>Start Time:</strong> {formatDateTime(scheduleData.start_time)}</p>
                <p><strong>End Time:</strong> {formatDateTime(scheduleData.end_time)}</p>
                <p><strong>Duration:</strong> {calculateDuration(scheduleData.start_time, scheduleData.end_time)}</p>
              </div>
            </div>


            {scheduleData.metadata && Object.keys(scheduleData.metadata).length > 0 && (
              <div style={{ marginTop: '15px' }}>
                <h4>Metadata</h4>
                <div style={{
                  border: '1px solid #eee',
                  borderRadius: '8px',
                  padding: '15px',
                  marginTop: '10px'
                }}>
                  <pre style={{
                    backgroundColor: '#fff',
                    padding: '10px',
                    borderRadius: '4px',
                    margin: 0
                  }}>
                    {JSON.stringify(scheduleData.metadata, null, 2)}
                  </pre>
                </div>
              </div>
            )}
           </div>
        );


      case 'tasks':
        return (
          <div style={{ marginTop: '15px' }}>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333' // Black color for header icon
              }}>
                assignment
              </span>
              Schedule Tasks
            </h4>
            {scheduleData.task_ids && scheduleData.task_ids.length > 0 ? (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <CommonTaskList
                  presetFilters={{ schedule_id: scheduleId }}
                  rowsPerPage={5}
                  initialSearchQuery=""
                />
              </div>
            ) : (
              <p>No tasks assigned to this schedule.</p>
            )}
          </div>
        );


      case 'workforce':
        return (
          <div style={{ marginTop: '15px' }}>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333' // Black color for header icon
              }}>
                engineering
              </span>
              Assigned Workforce
            </h4>
            {scheduleData.workforce_id ? (
              <div style={{
                backgroundColor: '#f9f9f9',
                padding: '15px',
                borderRadius: '8px',
                marginTop: '10px'
              }}>
                <p><strong>Workforce ID:</strong> {scheduleData.workforce_id}</p>
                <p><strong>Status:</strong> Active</p>
                <p><strong>Assigned On:</strong> {formatDateTime(scheduleData.assigned_date)}</p>
              </div>
            ) : (
              <p>No workforce assigned to this schedule.</p>
            )}
          </div>
        );


      default:
        return null;
    }
  };


  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading schedule details...</div>
      </Modal>
    );
  }


  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }


  const firstLetter = scheduleData.schedule_name?.charAt(0)?.toUpperCase() || '?';


  return (
    <Modal onClose={onClose}>
      {/* Header Section */}
      <div style={{
        display: 'flex',
        alignItems: 'center',
        padding: '10px 10px 10px',
        borderBottom: '1px solid #eee'
      }}>
      <div style={{
  width: '60px',
  height: '60px',
  borderRadius: '50%',
  backgroundColor: '#e0e0e0', // Changed to light grey
  color: '#000000', // Changed to black
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'center',
  fontSize: '20px',
  fontWeight: 'bold',
  marginRight: '16px'
}}>
          {firstLetter}
        </div>
        <div>
          <div style={{
            fontWeight: '600',
            fontSize: '20px',
            lineHeight: '1.4',
            color: 'black'
          }}>
            {scheduleData.schedule_name}
          </div>
          <div style={{
            fontSize: '16px',
            color: 'black',
            marginTop: '10px',
            fontWeight: 'bold'
          }}>
            ID: #{scheduleData.sch_id}
          </div>
        </div>
      </div>



      {/* Tabs */}
      <div style={{
        display: 'flex',
        borderBottom: '1px solid #e0e0e0',
        marginBottom: '20px',
        padding: '0 10px'
      }}>
        {TAB_CONFIG.map(tab => (
          <button
            key={tab.id}
            style={{
              padding: '12px 16px',
              border: 'none',
              backgroundColor: 'transparent',
              cursor: 'pointer',
              fontSize: '14px',
              fontWeight: activeTab === tab.id ? '400' : '400',
              color: activeTab === tab.id ? '#34c759' : '#333',
              position: 'relative',
              whiteSpace: 'nowrap',
              minWidth: activeTab === tab.id ? '100px' : '50px',
              textAlign: 'center',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              gap: '8px',
              transition: 'min-width 0.2s ease'
            }}
            onClick={() => setActiveTab(tab.id)}
            onMouseEnter={() => setHoveredTab(tab.id)}
            onMouseLeave={() => setHoveredTab(null)}
          >
            <span className="material-symbols-outlined" style={{
              color: activeTab === tab.id ? '#34c759' : '#333',
              fontSize: '20px'
            }}>
              {tab.icon}
            </span>
            {(activeTab === tab.id || hoveredTab === tab.id) && (
              <span>{tab.label}</span>
            )}
            {activeTab === tab.id && (
              <div style={{
                position: 'absolute',
                bottom: '-1px',
                left: '0',
                right: '0',
                height: '2px',
                backgroundColor: '#34c759'
              }} />
            )}
          </button>
        ))}
      </div>


      {/* Tab Content */}
      <div style={{
        maxHeight: 'calc(100vh - 200px)',
        overflowY: 'auto',
        padding: '0px 20px',
        marginBottom: '20px'
      }}>
        {renderTabContent()}
      </div>
    </Modal>
  );
};


export default ScheduleView;