import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import ScheduleView from './ScheduleView';
import ScheduleUpdate from './ScheduleUpdate';
import ScheduleDelete from './ScheduleDelete';
import CommonData from '../../common/CommonData';

const SCHEDULE_HEADER_TO_KEY_MAP = {
    "Schedule Name": {
        key: "schedule_name",
        label: "Schedule Name",
        sortable: true,
        clickable: true
    },
    "Description": { 
        key: "description", 
        label: "Description", 
        sortable: true, 
        clickable: false 
    },
    "Start Time": {
        key: "start_time",
        label: "Start Time",
        sortable: true,
        clickable: false
    },
    "End Time": {
        key: "end_time",
        label: "End Time",
        sortable: false,
        clickable: false
    }
};
const SchedulesList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const accountId = '67fec97f95809e100c9954d4';
  const [selectedSchedule, setSelectedSchedule] = useState(null);
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  // Get common enum values from your API via the CommonData hook
  const enums = CommonData();

  // Build searchFields dynamically for schedules
const searchFields = [
  { 
    name: 'q', 
    type: 'text', 
    placeholder: 'Search schedules...', 
    defaultValue: initialSearchQuery 
  },
  { 
    name: 'start_date_from', 
    type: 'date', 
    placeholder: 'From Date' 
  },
  { 
    name: 'start_date_to', 
    type: 'date', 
    placeholder: 'To Date' 
  },
  {
    name: 'status',
    type: 'dropdown',
    placeholder: 'Status',
    options: [
      { label: 'All', value: '' },
      { label: 'Active', value: 'active' },
      { label: 'Inactive', value: 'inactive' }
    ]
  }
];

  const fetchDataUrl = `/schedules/?account_id=${accountId}`;

  const actionFormats = [
    { label: "view", modal: true, url: "/schedules/view/{id}" },
    { label: "edit", modal: true, url: "/schedules/update/{id}" },
    { label: "delete", modal: true, url: "/schedules/delete/{id}" }
  ];

  const handleActionClick = (scheduleId, action) => {
    setSelectedSchedule({ 
      scheduleId, 
      modalType: action.label 
    });
  };

  const handleCellClick = (header, item) => {
    if (header === "Schedule Name") {
      setSelectedSchedule({
        scheduleId: item.schedule_id,
        modalType: "view"
      });
    }
  };

  const closeModal = () => setSelectedSchedule(null);
  const refreshList = () => setRefreshTrigger(prev => prev + 1);

  return (
    <div className="schedules-list-container">
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={SCHEDULE_HEADER_TO_KEY_MAP}
        dataKey="schedules"
        identifierKey="schedule_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}
        refreshTrigger={refreshTrigger}
        searchFields={searchFields}
        renderCell={(value, column) => {
          if (value === undefined || value === null) {
            return <span style={{ color: '#999' }}>None</span>;
          }
          return String(value);
        }}
        tableProps={{
          striped: true,
          hover: true,
          responsive: true
        }}
      />

      {selectedSchedule?.modalType === "view" && (
        <ScheduleView 
          scheduleId={selectedSchedule.scheduleId}
          onClose={closeModal}
        />
      )}

      {selectedSchedule?.modalType === "edit" && (
        <ScheduleUpdate 
          scheduleId={selectedSchedule.scheduleId}
          onClose={closeModal}
          onUpdateSuccess={refreshList}
        />
      )}

      {selectedSchedule?.modalType === "delete" && (
        <ScheduleDelete 
          scheduleId={selectedSchedule.scheduleId}
          onClose={closeModal}
          onDeleteSuccess={refreshList}
        />
      )}
    </div>
  );
};

export default SchedulesList;