import React, { useState, useMemo } from 'react';
import { withTheme } from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';

// Custom Widget Components
const TimePickerWidget = (props) => {
  const { value, onChange } = props;
  return (
    <input
      type="time"
      value={value || ''}
      onChange={(e) => onChange(e.target.value)}
      style={{
        padding: '8px 12px',
        border: '1px solid #ddd',
        borderRadius: '4px',
        width: '100%'
      }}
    />
  );
};

const DatePickerWidget = (props) => {
  const { value, onChange } = props;
  return (
    <input
      type="date"
      value={value || ''}
      onChange={(e) => onChange(e.target.value)}
      style={{
        padding: '8px 12px',
        border: '1px solid #ddd',
        borderRadius: '4px',
        width: '100%'
      }}
    />
  );
};

// Custom Recurrence Field Component
const RecurrenceField = (props) => {
  const { formData, onChange } = props;
  const [showCustom, setShowCustom] = useState(false);
  const [selectedDay, setSelectedDay] = useState('wed');

  const daysOfWeek = [
    { value: 'mon', label: 'Monday' },
    { value: 'tue', label: 'Tuesday' },
    { value: 'wed', label: 'Wednesday' },
    { value: 'thu', label: 'Thursday' },
    { value: 'fri', label: 'Friday' },
    { value: 'sat', label: 'Saturday' },
    { value: 'sun', label: 'Sunday' }
  ];

  const recurrenceOptions = [
    { value: 'none', label: 'Does not repeat' },
    { value: 'daily', label: 'Daily' },
    { value: 'weekly', label: `Weekly on ${daysOfWeek.find(d => d.value === selectedDay)?.label}` },
    { value: 'monthly', label: 'Monthly on the same date' },
    { value: 'weekdays', label: 'Every weekday (Monday to Friday)' },
    { value: 'annually', label: 'Annually on the same date' },
    { value: 'custom', label: 'Custom...' }
  ];

  const handleRecurrenceChange = (value) => {
    const newData = { 
      ...formData, 
      repeat_type: value,
      custom_recurrence: {
        ...formData.custom_recurrence,
        ...(value === 'weekdays' && { 
          days_of_week: ['mon', 'tue', 'wed', 'thu', 'fri'],
          frequency: 'week'
        }),
        ...(value === 'weekly' && {
          days_of_week: [selectedDay],
          frequency: 'week'
        })
      }
    };
    setShowCustom(value === 'custom');
    onChange(newData);
  };

  const handleCustomChange = (key, value) => {
    onChange({
      ...formData,
      custom_recurrence: {
        ...formData.custom_recurrence,
        [key]: value
      }
    });
  };

  return (
    <div style={{ marginTop: '16px' }}>
      {/* Basic Recurrence Selector */}
      <div style={{ marginBottom: '16px' }}>
        <select
          value={formData.repeat_type || 'none'}
          onChange={(e) => handleRecurrenceChange(e.target.value)}
          style={{
            width: '100%',
            padding: '8px 12px',
            border: '1px solid #ddd',
            borderRadius: '4px'
          }}
        >
          {recurrenceOptions.map(option => (
            <option key={option.value} value={option.value}>
              {option.label}
            </option>
          ))}
        </select>
      </div>

      {/* Day selector for weekly option */}
      {formData.repeat_type === 'weekly' && (
        <div style={{ marginBottom: '16px' }}>
          <label>Repeat on:</label>
          <select
            value={selectedDay}
            onChange={(e) => {
              setSelectedDay(e.target.value);
              handleCustomChange('days_of_week', [e.target.value]);
            }}
            style={{
              marginLeft: '8px',
              padding: '4px 8px',
              border: '1px solid #ddd',
              borderRadius: '4px'
            }}
          >
            {daysOfWeek.map(day => (
              <option key={day.value} value={day.value}>
                {day.label}
              </option>
            ))}
          </select>
        </div>
      )}

      {/* Custom Recurrence Panel */}
      {showCustom && (
        <div style={{
          padding: '16px',
          border: '1px solid #eee',
          borderRadius: '4px',
          backgroundColor: '#f9f9f9'
        }}>
          {/* Interval Selector */}
          <div style={{ marginBottom: '16px' }}>
            <div style={{ display: 'flex', alignItems: 'center', marginBottom: '8px' }}>
              <span style={{ marginRight: '8px' }}>Repeat every</span>
              <input
                type="number"
                min="1"
                value={formData.custom_recurrence?.interval || 1}
                onChange={(e) => handleCustomChange('interval', parseInt(e.target.value) || 1)}
                style={{
                  width: '60px',
                  padding: '4px 8px',
                  border: '1px solid #ddd',
                  borderRadius: '4px'
                }}
              />
              <select
                value={formData.custom_recurrence?.frequency || 'week'}
                onChange={(e) => handleCustomChange('frequency', e.target.value)}
                style={{
                  marginLeft: '8px',
                  padding: '4px 8px',
                  border: '1px solid #ddd',
                  borderRadius: '4px'
                }}
              >
                <option value="day">Day(s)</option>
                <option value="week">Week(s)</option>
                <option value="month">Month(s)</option>
                <option value="year">Year(s)</option>
              </select>
            </div>

            {/* Days of Week Selection */}
            {formData.custom_recurrence?.frequency === 'week' && (
              <div style={{ marginTop: '12px' }}>
                <div style={{ marginBottom: '8px' }}>Repeat on:</div>
                <div style={{ display: 'flex', gap: '12px', flexWrap: 'wrap' }}>
                  {daysOfWeek.map(day => (
                    <label key={day.value} style={{ display: 'flex', alignItems: 'center' }}>
                      <input
                        type="checkbox"
                        checked={formData.custom_recurrence?.days_of_week?.includes(day.value) || false}
                        onChange={(e) => {
                          const days = formData.custom_recurrence?.days_of_week || [];
                          const newDays = e.target.checked
                            ? [...days, day.value]
                            : days.filter(d => d !== day.value);
                          handleCustomChange('days_of_week', newDays);
                        }}
                        style={{ marginRight: '4px' }}
                      />
                      {day.label}
                    </label>
                  ))}
                </div>
              </div>
            )}
          </div>

          {/* End Conditions */}
          <div>
            <div style={{ marginBottom: '8px' }}>Ends:</div>
            <div style={{ display: 'flex', flexDirection: 'column', gap: '8px' }}>
              <label style={{ display: 'flex', alignItems: 'center' }}>
                <input
                  type="radio"
                  name="end_type"
                  value="never"
                  checked={formData.custom_recurrence?.end_type === 'never'}
                  onChange={() => handleCustomChange('end_type', 'never')}
                  style={{ marginRight: '8px' }}
                />
                Never
              </label>

              <div style={{ display: 'flex', alignItems: 'center' }}>
                <label style={{ display: 'flex', alignItems: 'center', marginRight: '8px' }}>
                  <input
                    type="radio"
                    name="end_type"
                    value="after"
                    checked={formData.custom_recurrence?.end_type === 'after'}
                    onChange={() => handleCustomChange('end_type', 'after')}
                    style={{ marginRight: '8px' }}
                  />
                  After
                </label>
                <input
                  type="number"
                  min="1"
                  value={formData.custom_recurrence?.occurrences || 10}
                  onChange={(e) => handleCustomChange('occurrences', parseInt(e.target.value) || 1)}
                  disabled={formData.custom_recurrence?.end_type !== 'after'}
                  style={{
                    width: '60px',
                    padding: '4px 8px',
                    border: '1px solid #ddd',
                    borderRadius: '4px',
                    marginRight: '8px'
                  }}
                />
                occurrences
              </div>

              <div style={{ display: 'flex', alignItems: 'center' }}>
                <label style={{ display: 'flex', alignItems: 'center', marginRight: '8px' }}>
                  <input
                    type="radio"
                    name="end_type"
                    value="on_date"
                    checked={formData.custom_recurrence?.end_type === 'on_date'}
                    onChange={() => handleCustomChange('end_type', 'on_date')}
                    style={{ marginRight: '8px' }}
                  />
                  On
                </label>
                <input
                  type="date"
                  value={formData.custom_recurrence?.end_date || ''}
                  onChange={(e) => handleCustomChange('end_date', e.target.value)}
                  disabled={formData.custom_recurrence?.end_type !== 'on_date'}
                  style={{
                    padding: '4px 8px',
                    border: '1px solid #ddd',
                    borderRadius: '4px'
                  }}
                />
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

// Main Form Component
const ScheduleForm = ({ initialData, onSubmit }) => {
  const initialFormData = useMemo(() => ({
    schedule_name: '',
    description: '',
    start_date: '',
    start_time: '',
    end_time: '',
    recurrence: {
      repeat_type: 'none',
      custom_recurrence: {
        interval: 1,
        frequency: 'week',
        days_of_week: ['wed'],
        end_type: 'never',
        occurrences: 10,
        end_date: ''
      }
    },
    ...initialData
  }), [initialData]);

  const [formData, setFormData] = useState(initialFormData);
  const [submitted, setSubmitted] = useState(false);

  const Form = useMemo(() => withTheme({
    widgets: { 
      TimePickerWidget,
      DatePickerWidget 
    },
    fields: { 
      recurrenceField: RecurrenceField 
    }
  }), []);

  const handleSubmit = ({ formData }) => {
    setSubmitted(true);
    
    // Validate required fields
    if (!formData.schedule_name) {
      alert('Schedule name is required');
      return;
    }
    
    if (!formData.start_date) {
      alert('Start date is required');
      return;
    }
    
    if (!formData.start_time || !formData.end_time) {
      alert('Both start and end times are required');
      return;
    }
    
    if (formData.start_time >= formData.end_time) {
      alert('End time must be after start time');
      return;
    }

    // Prepare final data with combined datetime
    const payload = {
      ...formData,
      start_datetime: `${formData.start_date}T${formData.start_time}`,
      end_datetime: `${formData.start_date}T${formData.end_time}`,
    };

    onSubmit(payload);
  };

  return (
    <div style={{ maxWidth: '600px', padding: '20px' }}>
      {/* <h2 style={{ marginBottom: '20px' }}>Create New Schedule</h2> */}
      
      <Form
        schema={{
          type: "object",
          required: ["schedule_name", "start_date", "start_time", "end_time"],
          properties: {
            schedule_name: { 
              type: "string", 
              title: "Schedule Name",
              minLength: 2,
              maxLength: 100
            },
            description: { 
              type: "string", 
              title: "Description" 
            },
            start_date: {
              type: "string",
              format: "date",
              title: "Start Date"
            },
            start_time: { 
              type: "string", 
              title: "Start Time" 
            },
            end_time: { 
              type: "string", 
              title: "End Time" 
            },
            recurrence: {
              type: "object",
              title: "Repeat"
            }
          }
        }}
        uiSchema={{
          "ui:order": [
            "schedule_name",
            "description",
            "start_date",
            "start_time",
            "end_time",
            "recurrence"
          ],
          schedule_name: {
            "ui:autofocus": true,
            "ui:placeholder": "Enter schedule name"
          },
          description: {
            "ui:widget": "textarea",
            "ui:options": {
              rows: 3
            }
          },
          start_date: {
            "ui:widget": "DatePickerWidget"
          },
          start_time: {
            "ui:widget": "TimePickerWidget"
          },
          end_time: {
            "ui:widget": "TimePickerWidget"
          },
          recurrence: {
            "ui:field": "recurrenceField"
          }
        }}
        formData={formData}
        onChange={({ formData }) => setFormData(formData)}
        onSubmit={handleSubmit}
        validator={validator}
        showErrorList={submitted}
        noHtml5Validate
      />
    </div>
  );
};

export default ScheduleForm;