import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import '../../themes/blue/Tasks.css';

const TAB_CONFIG = [
  { id: 'details', label: 'Rule Information', icon: 'assignment' },
  { id: 'pricing', label: 'Pricing Structure', icon: 'attach_money' },
  { id: 'conditions', label: 'Conditions', icon: 'rule' }
];

const PricingRulesView = ({ pricingRuleId, onClose }) => {
  const [pricingRuleData, setPricingRuleData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('details');
  const [hoveredTab, setHoveredTab] = useState(null);

  useEffect(() => {
    const fetchPricingRuleDetails = async () => {
      try {
        const res = await axios.get(`/pricingrules/${pricingRuleId}`);
        // Make sure we're accessing the data correctly
        setPricingRuleData({
          ...res.data,
          // Ensure pricing_rules_id is properly set
          pricing_rules_id: res.data.pricing_rules_id || pricingRuleId
        });
      } catch (err) {
        setError('Failed to load pricing rule details. Please try again.');
        console.error('Error fetching pricing rule:', err);
      } finally {
        setLoading(false);
      }
    };

    fetchPricingRuleDetails();
  }, [pricingRuleId]);

  const renderTabContent = () => {
    if (!pricingRuleData) return null;

    switch (activeTab) {
      case 'details':
        return (
          <div className="pricing-rule-details">
            <div className="detail-section">
              <h3>
                <span className="material-symbols-outlined" style={{
                  verticalAlign: 'middle',
                  marginRight: '8px',
                  color: '#333'
                }}>
                  assignment
                </span>
                Basic Information
              </h3>
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <p><strong>Rule Name:</strong> {pricingRuleData.rule_name}</p>
                <p><strong>Status:</strong> {pricingRuleData.status}</p>
                <p><strong>Created Date:</strong> {new Date(pricingRuleData.created_date).toLocaleString()}</p>
                <p><strong>Updated Date:</strong> {new Date(pricingRuleData.updated_date).toLocaleString()}</p>
              </div>
            </div>
          </div>
        );

      case 'pricing':
        return (
          <div>
            <h3>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                attach_money
              </span>
              Pricing Structure
            </h3>
            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <p><strong>Pricing Type:</strong> {pricingRuleData.pricing_type_value || pricingRuleData.pricing_type}</p>
              <p><strong>Base Cost:</strong> ${pricingRuleData.base_cost}</p>
              <p><strong>Rate:</strong> ${pricingRuleData.rate} per {pricingRuleData.unit}</p>
            </div>
          </div>
        );

      case 'conditions':
        return (
          <div>
            <h3>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                rule
              </span>
              Conditions
            </h3>
            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              {pricingRuleData.conditions && pricingRuleData.conditions.length > 0 ? (
                pricingRuleData.conditions.map((condition, index) => (
                  <div key={index} style={{
                    borderRadius: '6px',
                    padding: '0px',
                    marginBottom: '10px',
                    backgroundColor: '#fafafa'
                  }}>
                    <p><strong>Label:</strong> {condition.label}</p>
                    <p><strong>Applicable:</strong> {condition.applicable ? 'Yes' : 'No'}</p>
                    <p><strong>From Time:</strong> {condition.from_time}</p>
                    <p><strong>To Time:</strong> {condition.to_time}</p>
                    <p><strong>Calculation Type:</strong> {condition.calculation_type}</p>
                    <p><strong>Value:</strong> {condition.value}</p>
                  </div>
                ))
              ) : (
                <p>No conditions specified for this pricing rule</p>
              )}
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Modal title="Pricing Rule Details" onClose={onClose}>
        <div className="loading-indicator">Loading pricing rule details...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal title="Pricing Rule Details" onClose={onClose}>
        <div className="error-message">{error}</div>
      </Modal>
    );
  }

 return (
    <Modal onClose={onClose}>
      <div style={{ height: '100%', overflow: 'hidden' }}>
        {/* Header */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          gap: '20px',
          padding: '0px 10px 10px',
          borderBottom: '1px solid #e0e0e0'
        }}>
          <div style={{
            width: '60px',
            height: '60px',
            borderRadius: '50%',
            backgroundColor: '#e0e0e0',
            color: '#000',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            fontSize: '20px',
            fontWeight: 'bold'
          }}>
            {pricingRuleData?.rule_name?.charAt(0)?.toUpperCase() || 'P'}
          </div>
          <div>
            <h3 style={{ margin: '0px', fontWeight: 'bold', fontSize: '20px', color: 'black' }}>
              {pricingRuleData?.rule_name || 'Pricing Rule'}
            </h3>
<p style={{ marginTop: '5px', fontSize: '16px', color: 'black', fontWeight: 'bold' }}>
  ID: #{pricingRuleData?.pricing_rules_id || pricingRuleData?.pricing_rules_id}
</p>
          </div>
        </div>

        {/* Tabs */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '20px 6px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '2px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && <span>{tab.label}</span>}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-20px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Content */}
        <div style={{ padding: '0 20px', overflow: 'hidden' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default PricingRulesView;