import React, { useState, useEffect } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import PricingRulesView from './PricingRulesView';
import PricingRulesUpdate from './PricingRulesUpdate';
import PricingRulesDB from './PricingRulesDB';
import CommonData from '../../common/CommonData';

const PRICING_RULES_HEADER_TO_KEY_MAP = {
  "Rule Name": {
    key: "rule_name",
    label: "Rule Name",
    sortable: true,
    clickable: true
  },
  "Pricing Type": { key: "pricing_type_value", label: "Pricing Type", sortable: true },
  "Base Cost": { key: "base_cost", label: "Base Cost", sortable: true },
  "Rate": { key: "rate", label: "Rate", sortable: false },
  "Unit": { key: "unit", label: "Unit", sortable: false },
  "Created Date": { key: "created_date", label: "Created Date", sortable: false },
};

const PricingRulesList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  // Define initial state in a constant for easy reset
  const initialState = {
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery,
    created_date_from: '',
    created_date_to: '',
    status: ''
  };

  const [searchParams, setSearchParams] = useState(initialState);
  const [selectedPricingRule, setSelectedPricingRule] = useState(null);
  const [pricingTypes, setPricingTypes] = useState([]);
  const enums = CommonData();

  useEffect(() => {
    if (enums && enums.pricing_type) {
      setPricingTypes(enums.pricing_type.map(type => ({
        label: type.value,
        value: type.key
      })));
    }
  }, [enums]);

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search by rule name...', defaultValue: initialSearchQuery },
    { name: 'created_date_from', type: 'date', placeholder: 'From Date', value: searchParams.created_date_from },
    { name: 'created_date_to', type: 'date', placeholder: 'To Date', value: searchParams.created_date_to },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      value: searchParams.status,
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: '1' },
        { label: 'Inactive', value: '0' }
      ]
    }
  ];

  const transformResponse = (response) => {
    if (!response || !response.users) return { items: [], total: 0 };

    const transformedItems = response.users.map(item => ({
      ...item,
      id: item._id?.$oid || item.user_id,
      pricing_type_value: item.pricing_type_value || item.pricing_type,
      created_date: new Date(item.created_date).toLocaleString()
    }));

    return {
      items: transformedItems,
      total: response.total_count
    };
  };

  const actionFormats = [
    { label: "view", modal: true, url: "/pricingrules/view/{id}" },
    { label: "edit", modal: true, url: "/pricingrules/update/{id}" },
    { label: "delete", modal: true, url: "/pricingrules/delete/{id}" }
  ];

 const handleActionClick = (pricingRuleId, action) => {
  if (action.label === "edit") {
    setSelectedPricingRule({ 
      pricingRuleId, 
      modalType: "update"  // Make sure this matches your modal rendering condition
    });
  } else if (action.label === "view") {
    setSelectedPricingRule({
      pricingRuleId,
      modalType: "view"
    });
  } else if (action.label === "delete") {
    setSelectedPricingRule({
      pricingRuleId,
      modalType: "delete"
    });
  }
};

  const handleCellClick = (header, item) => {
    if (header === "Rule Name") {
      const ruleId = item.id || item._id?.$oid;
      if (ruleId) {
        setSelectedPricingRule({
          pricingRuleId: ruleId,
          modalType: "view"
        });
      }
    }
  };

  const handleSearch = (params) => {
    const fromDate = new Date(params.created_date_from);
    const toDate = new Date(params.created_date_to);

    // Validate: To Date must be greater than or equal to From Date
    if (params.created_date_from && params.created_date_to && toDate < fromDate) {
      alert("To Date must be greater than or equal to From Date.");
      return;
    }

    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      created_date_from: params.created_date_from || '',
      created_date_to: params.created_date_to || '',
      status: params.status || '',
      skip: 0 // Reset to first page on new search
    }));
  };

  // New function to handle clear filters
  const handleClearFilters = () => {
    setSearchParams(initialState);
  };

  const handlePageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage
    }));
  };

  const buildFetchUrl = () => {
    const baseUrl = `/pricingrules/list/all/?skip=${searchParams.skip}&limit=${searchParams.limit}`;
    const params = new URLSearchParams();
    
    if (searchParams.q) params.append('q', searchParams.q);
    if (searchParams.created_date_from) params.append('from', searchParams.created_date_from);
    if (searchParams.created_date_to) params.append('to', searchParams.created_date_to);
    if (searchParams.status) params.append('is_active', searchParams.status);

    return `${baseUrl}&${params.toString()}`;
  };

  const closeModal = () => {
    setSelectedPricingRule(null);
  };

  

  return (
    <div style={{ textAlign: 'left', paddingLeft: '10px' }}>
      <SearchPage
        fetchDataUrl={buildFetchUrl()}
        ListComponent={List}
        itemKeyMapping={PRICING_RULES_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}
        transformResponse={transformResponse}
        searchFields={searchFields}
        onSearch={handleSearch}
        onClearFilters={handleClearFilters} // Pass the clear filters handler
        onPageChange={handlePageChange}
      />

      {selectedPricingRule?.modalType === "view" && (
        <PricingRulesView
          pricingRuleId={selectedPricingRule.pricingRuleId}
          onClose={closeModal}
        />
      )}

     {selectedPricingRule?.modalType === "update" && (
    <PricingRulesUpdate
        pricingRuleId={selectedPricingRule.pricingRuleId}
        onClose={closeModal}
        onUpdateSuccess={() => {
            closeModal();
            // This will trigger a refetch of the data
            setSearchParams(prev => ({ ...prev }));
        }}
    />
)}

      {selectedPricingRule?.modalType === "delete" && (
        <PricingRulesDB
          pricingRuleId={selectedPricingRule.pricingRuleId}
          onClose={closeModal}
          onDeleteSuccess={() => {
  setSelectedPricingRule(null); // Close modal
  setSearchParams(prev => ({ ...prev })); // Trigger refetch
}}
        />
      )}
    </div>
  );
};

export default PricingRulesList;