import React, { useState, useEffect } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';
import { useNavigate } from 'react-router-dom';

// Pricing Rule Schema Definition
export const pricingRuleSchema = {
    type: "object",
    required: ["rule_name", "pricing_type", "base_cost", "rate", "unit"],
    properties: {
        rule_name: { 
            type: "string", 
            title: "Rule Name",
            minLength: 3,
            maxLength: 50
        },
        pricing_type: { 
            type: "string", 
            title: "Select a Pricing Type",
            enum: [],
            enumNames: []
        },
        base_cost: { 
            type: "number", 
            title: "Base Cost",
            minimum: 0
        },
        rate: { 
            type: "number", 
            title: "Rate",
            minimum: 0
        },
        unit: { 
            type: "string", 
            title: "Unit",
            enum: ["KM", "Minutes"],
            enumNames: ["Kilometers (KM)", "Minutes"] 
        },
        conditions: {
            type: "array",
            title: "Time Conditions",
            items: {
                type: "object",
                required: ["label", "from_time", "to_time", "calculation_type", "value"],
                properties: {
                    label: {
                        type: "string",
                        title: "Label",
                        minLength: 2,
                        maxLength: 30
                    },
                    from_time: {
                        type: "string",
                        format: "time",
                        title: "From Time"
                    },
                    to_time: {
                        type: "string",
                        format: "time",
                        title: "To Time"
                    },
                    calculation_type: {
                        type: "string",
                        title: "Calculation Type",
                        enum: ["amount", "percentage"],
                        enumNames: ["Fixed Amount", "Percentage"]
                    },
                    value: {
                        type: "number",
                        title: "Value",
                        minimum: 0
                    }
                }
            }
        }
    }
};

// Custom Array Field Template for Conditions
function ArrayFieldTemplate(props) {
    return (
        <div>
            <div style={{ marginBottom: '10px' }}>
                {props.title && <h5>{props.title}</h5>}
                {props.description && <p>{props.description}</p>}
            </div>
            
            {props.items.map((element, index) => (
                <div key={index} style={{ 
                    marginBottom: '20px', 
                    padding: '15px',
                    border: '1px solid #eee',
                    borderRadius: '4px',
                    position: 'relative'
                }}>
                    {element.children}
                    
                    {element.hasRemove && (
                        <button
                            type="button"
                            onClick={element.onDropIndexClick(element.index)}
                            style={{
                                position: 'absolute',
                                right: '10px',
                                top: '10px',
                                width: '30px',
                                height: '30px',
                                padding: '0',
                                backgroundColor: '#ff4444',
                                color: 'white',
                                border: 'none',
                                borderRadius: '4px',
                                cursor: 'pointer',
                                fontSize: '16px',
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'center',
                                transition: 'background-color 0.3s'
                            }}
                            onMouseOver={(e) => e.currentTarget.style.backgroundColor = '#cc0000'}
                            onMouseOut={(e) => e.currentTarget.style.backgroundColor = '#ff4444'}
                            title="Remove condition"
                        >
                            ×
                        </button>
                    )}
                </div>
            ))}
            
            {props.canAdd && (
                <button
                    type="button"
                    onClick={props.onAddClick}
                    style={{
                        backgroundColor: '#f0f0f0',
                        color: '#00D563',
                        padding: '8px 16px',
                        border: '1px solid white',
                        borderRadius: '4px',
                        cursor: 'pointer',
                        marginTop: '10px',
                        fontSize: '14px',
                        fontWeight: 'bold',
                        transition: 'all 0.3s'
                    }}
                    onMouseOver={(e) => {
                        e.currentTarget.style.backgroundColor = '#00D563';
                        e.currentTarget.style.color = 'white';
                    }}
                    onMouseOut={(e) => {
                        e.currentTarget.style.backgroundColor = '#f0f0f0';
                        e.currentTarget.style.color = '#00D563';
                    }}
                >
                    + Add Time Condition
                </button>
            )}
        </div>
    );
}

// UI Schema Configuration
export const pricingRuleUiSchema = {
    pricing_type: {
        "ui:widget": "select",
        "ui:options": {
            label: true
        }
    },
    unit: {
        "ui:widget": "radio",
        "ui:options": {
            inline: true
        }
    },
    base_cost: {
        "ui:widget": "updown"
    },
    rate: {
        "ui:widget": "updown"
    },
    conditions: {
        "ui:options": {
            orderable: false
        },
        "ui:ArrayFieldTemplate": ArrayFieldTemplate,
        items: {
            "ui:field": "fieldset",
            "ui:options": {
                collapsible: true,
                collapsed: false
            },
            calculation_type: {
                "ui:widget": "radio",
                "ui:options": {
                    inline: true
                }
            },
            from_time: {
                "ui:widget": "time"
            },
            to_time: {
                "ui:widget": "time"
            }
        }
    },
    "ui:submitButtonOptions": {
        "props": {
            className: "custom-submit-button"
        },
        "norender": false,
        "submitText": "Submit"
    }
};

// Main Form Component
const PricingRulesForm = ({ initialData, onSubmit }) => {
    const [formData, setFormData] = useState(initialData || {});
    const [schema, setSchema] = useState(pricingRuleSchema);
    const [pricingTypeOptions, setPricingTypeOptions] = useState([]);
    const [statusMessage, setStatusMessage] = useState({ text: '', type: '' });
    const [isSubmitting, setIsSubmitting] = useState(false);
    const navigate = useNavigate();

    // Check for duplicate rule names
    const checkForDuplicateRule = async (ruleName) => {
        try {
            const response = await axios.get(
                `/pricingrules/list/all/?q=${encodeURIComponent(ruleName)}`
            );
            
            return response.data.users?.some(rule => {
                const isSameName = rule.rule_name.toLowerCase() === ruleName.toLowerCase();
                const isDifferentRule = !initialData || rule._id?.$oid !== initialData._id?.$oid;
                return isSameName && isDifferentRule;
            });
        } catch (error) {
            console.error('Error checking for duplicate rules:', error);
            return false;
        }
    };

    // Load pricing types on component mount
    useEffect(() => {
        const loadPricingTypes = async () => {
            try {
                const res = await axios.get('crud/pricing_types?skip=0&limit=10&q=');
                const pricingTypes = res.data.items.map(item => ({
                    value: item.pricing_types_id.toString(),
                    label: item.pricing_type,
                    name: item.pricing_type
                }));

                setPricingTypeOptions(pricingTypes);

                const updatedSchema = {
                    ...pricingRuleSchema,
                    properties: {
                        ...pricingRuleSchema.properties,
                        pricing_type: {
                            ...pricingRuleSchema.properties.pricing_type,
                            enum: pricingTypes.map(t => t.value),
                            enumNames: pricingTypes.map(t => t.label)
                        }
                    }
                };
                
                setSchema(updatedSchema);

                if (initialData && initialData.pricing_type_value) {
                    const selectedType = pricingTypes.find(
                        type => type.name === initialData.pricing_type_value
                    );
                    if (selectedType) {
                        setFormData({
                            ...initialData,
                            pricing_type: selectedType.value,
                            conditions: initialData.conditions || []
                        });
                    }
                }
            } catch (err) {
                console.error('Error fetching pricing types:', err);
                setStatusMessage({
                    text: 'Failed to load pricing types. Please refresh the page.',
                    type: 'error'
                });
            }
        };

        loadPricingTypes();
    }, [initialData]);

    // Handle form submission
    const handleSubmit = async ({ formData }) => {
        setIsSubmitting(true);
        setStatusMessage({ text: '', type: '' });

        try {
            // Validate rule name
            if (!formData.rule_name || formData.rule_name.trim().length < 3) {
                throw new Error('Rule name must be at least 3 characters long');
            }

            // Check for duplicate rule name
            const isDuplicate = await checkForDuplicateRule(formData.rule_name);
            if (isDuplicate) {
                throw new Error('A pricing rule with this name already exists');
            }

            const selectedPricingType = pricingTypeOptions.find(
                option => option.value === formData.pricing_type
            );

            // Prepare payload
            const payload = {
                rule_name: formData.rule_name.trim(),
                pricing_type: formData.pricing_type,
                pricing_type_value: selectedPricingType?.name || "",
                base_cost: parseFloat(formData.base_cost),
                rate: parseFloat(formData.rate),
                unit: formData.unit,
                conditions: formData.conditions ? formData.conditions.map(condition => ({
                    label: condition.label.trim(),
                    from_time: condition.from_time,
                    to_time: condition.to_time,
                    calculation_type: condition.calculation_type,
                    value: parseFloat(condition.value)
                })) : [],
                account_id: localStorage.getItem('account_id'),
                user_id: localStorage.getItem('user_id')
            };
            
            // Submit the form
            await onSubmit(payload);
            
            // Show success message and redirect
            // setStatusMessage({
            //     text: `Pricing rule ${initialData ? 'updated' : 'created'} successfully!`,
            //     type: 'success'
            // });
            
            // Redirect after 1.5 seconds
            setTimeout(() => navigate('/pricingrules/'), 1500);
            
        } catch (error) {
            console.error('Error submitting form:', error);
            setStatusMessage({
                text: error.message || 'Error submitting form. Please try again.',
                type: 'error'
            });
        } finally {
            setIsSubmitting(false);
        }
    };

    return (
<div style={{ maxWidth: "800px", padding: '20px', marginLeft: '0' }}>

            {/* Status Message */}
            {statusMessage.text && (
                <div 
                    style={{ 
                        padding: '10px 15px',
                        marginBottom: '20px',
                        borderRadius: '4px',
                        backgroundColor: statusMessage.type === 'error' ? '#ffebee' : '#e8f5e9',
                        color: statusMessage.type === 'error' ? '#c62828' : '#2e7d32',
                        border: `1px solid ${statusMessage.type === 'error' ? '#ef9a9a' : '#a5d6a7'}`,
                        fontWeight: 'bold'
                    }}
                >
                    {statusMessage.text}
                </div>
            )}

            {/* Form Styles */}
            <style>
                {`
                .custom-submit-button {
                    background-color: #f0f0f0;
                    color: #00D563;
                    border: 1px solid white;
                    padding: 10px 20px;
                    border-radius: 4px;
                    cursor: pointer;
                    font-size: 14px;
                    font-weight: bold;
                    transition: all 0.3s;
                    margin-top: 20px;
                }
                .custom-submit-button:hover {
                    background-color: #00D563;
                    color: white;
                }
                .custom-submit-button:disabled {
                    background-color: #e0e0e0;
                    color: #9e9e9e;
                    cursor: not-allowed;
                }
                `}
            </style>

            {/* Main Form */}
            <Form
                schema={schema}
                uiSchema={{
                    ...pricingRuleUiSchema,
                    "ui:submitButtonOptions": {
                        ...pricingRuleUiSchema["ui:submitButtonOptions"],
                        submitText: isSubmitting ? 'Processing...' : (initialData ? 'Update Rule' : 'Submit'),
                        props: {
                            ...pricingRuleUiSchema["ui:submitButtonOptions"].props,
                            disabled: isSubmitting
                        }
                    }
                }}
                formData={formData}
                onChange={({ formData }) => setFormData(formData)}
                onSubmit={handleSubmit}
                validator={validator}
                showErrorList={false}
                disabled={isSubmitting}
            />
        </div>
    );
};

export default PricingRulesForm;