import React, { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import axios from 'axios';
import GoogleSignIn from '../Signup/GoogleSignIn';
import { storeUserData } from '../Login/AuthStorage';
import { useDispatch } from 'react-redux';

const Login = () => {
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [errorMessage, setErrorMessage] = useState('');
  const navigate = useNavigate();
  const dispatch = useDispatch();

  const handleLogin = async (e) => {
    e.preventDefault();

    if (!isValidEmail()) {
      setErrorMessage('Your email is invalid!');
      return;
    }

    if (!isPasswordValid()) {
      setErrorMessage('Password must be at least 8 characters long.');
      return;
    }

    try {
      const response = await axios.post('/auth/', { email, password });

      if (response.request.status === 200) {
        const userData = response.data.user;
        const roleId = parseInt(userData.roles);

        // Save user data + token to localStorage and Redux
        storeUserData({ ...userData, token: response.data.token }, dispatch);
        axios.defaults.withCredentials = true;

        console.log('Login successful:', userData);

        // Define redirection logic based on role and account
        let target = '';
        if (!userData.account_id) {
          target = '/setup';
        } else {
          switch (roleId) {
            case 1: // Superadmin
            case 2: // Sales/Accounts
              target = '/saas/dash';
              break;
            case 3: // Partner
              target = '/saas/partners'; // or another suitable route
              break;
            case 4: // Partner - Sales
              target = '/saas/partners'; // or another suitable route
              break;
            case 5: // Tech Team
              target = '/saas/tickets';
              break;
            case 100: // Account Owner
            case 101: // Manager
            case 102: // Finance
            case 103: // Workforce
            case 104: // Customer
            case 105: // Vendor
              target = '/dash';
              break;
            default:
              target = '/billing';
              break;
          }
        }

        console.log('Redirecting to:', target);
        navigate(target);
      } else {
        setErrorMessage('Validation failed. Please try again.');
      }
    } catch (error) {
      console.error('Error during login:', error);
      setErrorMessage('Login failed! Please try again.');
    }
  };

  const isPasswordValid = () => password.length >= 8;
  const isValidEmail = () =>
    /^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,4}$/.test(email);

  return (
    <div className="container">
      <h2>Welcome Back!</h2>
      <p className="note">
        Great to see you again! Sign in to continue where you left off and explore more.
      </p>
      <form onSubmit={handleLogin}>
        <input
          type="email"
          placeholder="Email"
          value={email}
          onChange={(e) => setEmail(e.target.value)}
        />
        <input
          type="password"
          placeholder="Password"
          value={password}
          onChange={(e) => setPassword(e.target.value)}
          minLength="8"
        />
        {errorMessage && <p style={{ color: 'red' }}>{errorMessage}</p>}
        <button type="submit" className="button">Login</button>
      </form>
      <p>
        Don't have an account? <Link to="/signup">Sign Up</Link>
      </p>
      <div className="orContainer">
        <div className="line"></div>
        <p className="orText">OR</p>
        <div className="line"></div>
      </div>
      <GoogleSignIn />
      <div style={{ marginTop: '20px' }}>
        <p>Please read more about our policies</p>
        <Link to="/terms">Terms of Service</Link> &nbsp; &amp; &nbsp;
        <Link to="/privacy">Privacy Policy</Link>
      </div>
    </div>
  );
};

export default Login;
