import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import '../../themes/blue/Tasks.css';


const TAB_CONFIG = [
  { id: 'tasks', label: 'Task Information', icon: 'assignment' },
  { id: 'schedules', label: 'Schedules', icon: 'schedule' },
  { id: 'routes', label: 'Routes', icon: 'route' },
  { id: 'customer', label: 'Customer', icon: 'person' },
  { id: 'resources', label: 'Resources', icon: 'engineering' }
];


const TaskView = ({ taskId, onClose }) => {
  const [taskData, setTaskData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('tasks');
  const [hoveredTab, setHoveredTab] = useState(null);


  useEffect(() => {
    const fetchTaskDetails = async () => {
      try {
        const res = await axios.get(`/tasks/${taskId}`);
        setTaskData(res.data);
      } catch (err) {
        setError('Failed to load task details. Please try again.');
        console.error('Error fetching task:', err);
      } finally {
        setLoading(false);
      }
    };


    fetchTaskDetails();
  }, [taskId]);


  const renderTabContent = () => {
    if (!taskData) return null;


    switch (activeTab) {
      case 'tasks':
        return (
          <div className="task-details">
            <div className="detail-section">
              <h3>
                <span className="material-symbols-outlined" style={{
                  verticalAlign: 'middle',
                  marginRight: '8px',
                  color: '#333'
                }}>
                  assignment
                </span>
                Basic Information
              </h3>
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                {/* <p><strong>Service:</strong> {taskData.service_title}</p> */}
                <p><strong>Task Mode:</strong> {taskData.task_mode}</p>
                <p><strong>Priority:</strong> {taskData.priority}</p>
                <p><strong>Capacity:</strong> {taskData.capacity}</p>
                <p><strong>Round Trip:</strong> {taskData.roundtrip ? 'Yes' : 'No'}</p>
                {/* <p><strong>Completion Deadline:</strong> {new Date(taskData.end_time).toLocaleString()}</p> */}
                <p><strong>Created Date:</strong> {new Date(taskData.created_date).toLocaleString()}</p>
              </div>
            </div>


            <div className="detail-section">
              <h3>
                <span className="material-symbols-outlined" style={{
                  verticalAlign: 'middle',
                  marginRight: '8px',
                  color: '#333'
                }}>
                  timelapse
                </span>
                Time Tracking
              </h3>
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <p><strong>Actual Start:</strong> {taskData.actual_start ? new Date(taskData.actual_start).toLocaleString() : 'Not started'}</p>
                <p><strong>Actual End:</strong> {taskData.actual_end ? new Date(taskData.actual_end).toLocaleString() : 'Not completed'}</p>
                <p><strong>Time Spent:</strong> {taskData.time_spent} minutes</p>
                <p><strong>Distance Travelled:</strong> {taskData.distance_travelled} km</p>
                <p><strong>Cost:</strong> ${taskData.cost}</p>
              </div>
            </div>


            <div className="detail-section">
              <h3>
                <span className="material-symbols-outlined" style={{
                  verticalAlign: 'middle',
                  marginRight: '8px',
                  color: '#333'
                }}>
                  checklist
                </span>
                Task Details
              </h3>
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <p><strong>Additional Services:</strong> {Array.isArray(taskData.addons) ? taskData.addons.join(', ') : 'None selected'}</p>
                <p><strong>Checklist Items:</strong> {Array.isArray(taskData.checklist) ? taskData.checklist.join(', ') : 'None selected'}</p>
              </div>
            </div>
          </div>
        );


      case 'schedules':
        return (
          <div>
            <h3>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                schedule
              </span>
              Scheduling Information
            </h3>
            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              {/* <p><strong>Scheduled Start:</strong> {new Date(taskData.start_time).toLocaleString()}</p>
              <p><strong>Scheduled End:</strong> {new Date(taskData.end_time).toLocaleString()}</p> */}
              <p><strong>Recurrence:</strong> {taskData.recurrence || 'One-time'}</p>
              <p><strong>Status:</strong> {taskData.status}</p>
             
              {/* Placeholder for live tracking */}
              <div style={{ marginTop: '15px', textAlign: 'center', padding: '20px', backgroundColor: '#f5f5f5', borderRadius: '8px' }}>
                <p>Live tracking visualization would appear here</p>
              </div>
            </div>
          </div>
        );


      case 'routes':
        return (
          <div>
            <h3>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                route
              </span>
              Route Information
            </h3>
            {Array.isArray(taskData.stops) && taskData.stops.length > 0 ? (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '15px' }}>
                  <div>
                    <h4>Waypoints</h4>
                    <div className="stops-list">
                      {taskData.stops
                        .sort((a, b) => a.sequence - b.sequence)
                        .map((stop, index) => (
                          <div key={index} style={{
                            padding: '10px',
                            borderBottom: index < taskData.stops.length - 1 ? '1px solid #eee' : 'none'
                          }}>
                            <p><strong>Stop #{stop.sequence}:</strong> {stop.type}</p>
                            <p><strong>Location:</strong> {stop.location}</p>
                            {stop.timing_details && (
                              <p><strong>Notes:</strong> {stop.timing_details}</p>
                            )}
                          </div>
                        ))}
                    </div>
                  </div>
                  <div>
                    <h4>Map View</h4>
                    <div style={{
                      height: '300px',
                      backgroundColor: '#f5f5f5',
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'center',
                      borderRadius: '8px'
                    }}>
                      <p>Map visualization would appear here</p>
                    </div>
                  </div>
                </div>
              </div>
            ) : (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '20px',
                marginTop: '10px',
                textAlign: 'center',
                color: '#666'
              }}>
                No route information available for this task
              </div>
            )}
          </div>
        );


      case 'customer':
        return (
          <div>
            <h3>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                person
              </span>
              Customer Information
            </h3>
            {taskData.customer_info ? (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '15px' }}>
                  <div>
                    <p><strong>Name:</strong> {taskData.customer_info.name}</p>
                    <p><strong>Contact:</strong> {taskData.customer_info.phone}</p>
                    <p><strong>Email:</strong> {taskData.customer_info.email || 'N/A'}</p>
                  </div>
                  <div>
                    <p><strong>Customer ID:</strong> {taskData.customer_info.id}</p>
                    <p><strong>Type:</strong> {taskData.customer_info.type || 'Standard'}</p>
                    <p><strong>Status:</strong> {taskData.customer_info.status || 'Active'}</p>
                  </div>
                </div>
              </div>
            ) : (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '20px',
                marginTop: '10px',
                textAlign: 'center',
                color: '#666'
              }}>
                No customer information available
              </div>
            )}
          </div>
        );


      case 'resources':
        return (
          <div>
            <h3>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333'
              }}>
                engineering
              </span>
              Assigned Resources
            </h3>
            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '15px' }}>
                <div>
                  <h4>Workforce</h4>
                  {taskData.assigned_workers?.length > 0 ? (
                    taskData.assigned_workers.map((worker, index) => (
                      <div key={index} style={{
                        padding: '10px',
                        borderBottom: index < taskData.assigned_workers.length - 1 ? '1px solid #eee' : 'none'
                      }}>
                        <p><strong>{worker.name}</strong> ({worker.role})</p>
                        <p>Contact: {worker.contact}</p>
                        <p>Status: {worker.status}</p>
                      </div>
                    ))
                  ) : (
                    <p>No workforce assigned</p>
                  )}
                </div>
                <div>
                  <h4>Vehicle</h4>
                  {taskData.assigned_vehicle ? (
                    <div>
                      <p><strong>Type:</strong> {taskData.assigned_vehicle.type}</p>
                      <p><strong>ID:</strong> {taskData.assigned_vehicle.id}</p>
                      <p><strong>Status:</strong> {taskData.assigned_vehicle.status}</p>
                      <p><strong>Capacity:</strong> {taskData.assigned_vehicle.capacity}</p>
                    </div>
                  ) : (
                    <p>No vehicle assigned</p>
                  )}
                </div>
              </div>
            </div>
          </div>
        );


      default:
        return null;
    }
  };


  if (loading) {
    return (
      <Modal title="Task Details" onClose={onClose}>
        <div className="loading-indicator">Loading task details...</div>
      </Modal>
    );
  }


  if (error) {
    return (
      <Modal title="Task Details" onClose={onClose}>
        <div className="error-message">{error}</div>
      </Modal>
    );
  }


  return (
    <Modal  onClose={onClose}>
      <div style={{ height: '100%', overflow: 'hidden' }}>
        {/* Header */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          gap: '20px',
          padding: '0px 10px 10px',
          borderBottom: '1px solid #e0e0e0'
        }}>
          <div style={{
            width: '60px',
            height: '60px',
            borderRadius: '50%',
            backgroundColor: '#e0e0e0',
            color: '#000',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            fontSize: '20px',
            fontWeight: 'bold'
          }}>
            {taskData.service_title?.charAt(0)?.toUpperCase() || 'T'}
          </div>
          <div>
            
            <p style={{ marginTop: '5px', fontSize: '16px', color: 'black', fontWeight: 'bold' }}>
               {taskData.task_mode}

              <h3 style={{ margin: '0px', fontWeight: 'bold', fontSize: '16px', color: 'black' }}>
              {taskData.service_title || 'ID'} #{taskData.t_id}
            </h3>
            </p>
          </div>
        </div>


        {/* Tabs */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '20px +6px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '2px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && <span>{tab.label}</span>}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-20px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>


        {/* Content */}
        <div style={{ padding: '0 20px', overflow: 'hidden' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};


export default TaskView;