import React, { useState, useEffect } from 'react';
import axios from 'axios';
import TaskForm from './TaskForm';
import Modal from '../../Layouts/Modal';

const TaskUpdate = ({ taskId, onClose, onUpdateSuccess }) => {
    const [taskData, setTaskData] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState('');

    useEffect(() => {
        const fetchTaskData = async () => {
            try {
                const res = await axios.get(`/tasks/${taskId}`);
                setTaskData(res.data);
                setLoading(false);
            } catch (err) {
                setError('Error fetching task data.');
                setLoading(false);
            }
        };

        fetchTaskData();
    }, [taskId]);

    const transformStops = (formData) => {
        const task_stops = [];
        let sequence = 1;

        if (formData.pickup_location) {
            task_stops.push({
                location: formData.pickup_location,
                type: "pickup",
                sequence: sequence++,
                timing: null
            });
        }

        if (formData.stops?.length) {
            formData.stops.forEach(stop => {
                task_stops.push({
                    location: stop.location,
                    type: "intermediate",
                    sequence: sequence++,
                    timing: stop.timing_details || null
                });
            });
        }

        if (formData.drop_location) {
            task_stops.push({
                location: formData.drop_location,
                type: "dropoff",
                sequence: sequence,
                timing: null
            });
        }

        return task_stops;
    };

    const handleSubmit = async (formData) => {
        try {
            const userData = JSON.parse(localStorage.getItem('userData'));
            const account_id = userData?.account_id;

            if (!account_id) {
                throw new Error('User account information not found');
            }

            const payload = {
                account_id: account_id,
                service_id: formData.service,
                task_mode: formData.task_mode,
                priority: formData.priority,
                roundtrip: formData.roundtrip,
                capacity: formData.capacity,
                scheduled_start: formData.task_start_date_time,
                scheduled_end: formData.task_end_date_time,
                actual_start: taskData?.actual_start || null,
                actual_end: taskData?.actual_end || null,
                time_spent: taskData?.time_spent || 0,
                distance_travelled: taskData?.distance_travelled || 0.0,
                cost: taskData?.cost || 0.0,
                addons: formData.addons || [],
                checklist: formData.checklist || [],
                variables: taskData?.variables || {},
                task_stops: transformStops(formData)
            };

            const res = await axios.put(`/tasks/${taskId}`, payload);
            alert("Task updated successfully!");
            onUpdateSuccess?.(res.data);
            onClose?.();
        } catch (err) {
            alert("Error updating task: " + (err.response?.data?.message || err.message));
        }
    };

    return (
        <Modal title="Edit Task" onClose={onClose}>
            {loading ? (
                <div>Loading task data...</div>
            ) : error ? (
                <div style={{ color: 'red' }}>{error}</div>
            ) : (
                <TaskForm 
                    initialData={taskData} 
                    onSubmit={handleSubmit} 
                    isUpdate={true}
                />
            )}
        </Modal>
    );
};

export default TaskUpdate;