import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import TaskView from './TaskView';
import TaskUpdate from './TaskUpdate';
import TaskDB from './TaskDB';
import '../../themes/blue/Tasks.css';

const TASKS_HEADER_TO_KEY_MAP = {
  "Task Mode": {
    key: "task_mode",
    label: "Task Mode",
    sortable: true,
    clickable: true // ✅ Make this field clickable
  },
  "Priority": { key: "priority", label: "Priority", sortable: true },
  "Capacity": { key: "capacity", label: "Capacity", sortable: true },
  "Round Trip": { key: "roundtrip", label: "Round Trip", sortable: false },
  "Created Date": { key: "created_date", label: "Created Date", sortable: false },
};

const TasksList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const account_id = localStorage.getItem('account_id');
  const fetchDataUrl = `/tasks/?account_id=${account_id}`;

  const [selectedTask, setSelectedTask] = useState(null);

  const actionFormats = [
    { label: "view", modal: true, url: "/tasks/view/{id}", icon: "visibility" },
    { label: "edit", modal: true, url: "/tasks/update/{id}", icon: "edit" },
    { label: "delete", modal: true, url: "/tasks/delete/{id}", icon: "delete", color: "error" }
  ];

  const handleActionClick = (taskId, action) => {
    if (["View", "Edit", "Delete"].includes(action.label)) {
      setSelectedTask({ taskId, modalType: action.label.toLowerCase() });
    }
  };

  // ✅ Handle cell click for "Task Mode"
  const handleCellClick = (header, item) => {
    if (header === "Task Mode") {
      const taskId = item.task_id || item.id;
      if (taskId) {
        console.log("✅ Task Mode clicked → Opening view modal for ID:", taskId);
        setSelectedTask({
          taskId,
          modalType: "view"
        });
      } else {
        console.warn("⚠️ Missing task_id in item:", item);
      }
    }
  };

  const closeModal = () => setSelectedTask(null);
  const handleUpdateSuccess = () => closeModal();
  const handleDeleteSuccess = () => closeModal();

  return (
    <div className="tasks-list-container">
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={TASKS_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="task_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick} // ✅ Add clickable handler
        searchPlaceholder="Search tasks by priority or mode"
      />

      {selectedTask?.modalType === "view" && (
        <TaskView taskId={selectedTask.taskId} onClose={closeModal} />
      )}
      {selectedTask?.modalType === "update" && (
        <TaskUpdate taskId={selectedTask.taskId} onClose={closeModal} onUpdateSuccess={handleUpdateSuccess} />
      )}
      {selectedTask?.modalType === "delete" && (
        <TaskDB taskId={selectedTask.taskId} onClose={closeModal} onDeleteSuccess={handleDeleteSuccess} />
      )}
    </div>
  );
};

export default TasksList;
