import React, { useState, useEffect, useRef, createContext, useContext } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';
import Script from 'react-load-script';

// Create context to manage dropdown state
const DropdownContext = createContext();

const taskSchema = {
  title: "Task Details",
  type: "object",
  required: ["service", "task_mode", "priority", "task_start_date_time", "task_end_date_time"],
  properties: {
    customer_id: {
      type: "string",
      title: "Customer",
    },
    pickup_location: {
      type: "string",
      title: "Pickup Location",
      minLength: 3
    },
    drop_location: {
      type: "string",
      title: "Drop Location",
      minLength: 3
    },
    service: {
      type: "string",
      title: "Service Type",
    },
    task_mode: {
      type: "string",
      title: "Task Mode",
      enum: ["scheduled", "on_demand", "multi_stop", "repeat"],
      enumNames: ["Scheduled", "On Demand", "Multi Stop", "Repeat"]
    },
    priority: {
      type: "string",
      title: "Priority Level",
      enum: ["low", "normal", "high", "urgent"],
      enumNames: ["Low", "Normal", "High", "Urgent"]
    },
    capacity: {
      type: "number",
      title: "Task Capacity",
      minimum: 1
    },
    roundtrip: {
      type: "boolean",
      title: "Round Trip Required"
    },
    task_start_date_time: {
      type: "string",
      format: "date-time",
      title: "Start Date & Time"
    },
    task_end_date_time: {
      type: "string",
      format: "date-time",
      title: "End Date & Time"
    },
    addons: {
      type: "array",
      title: "Additional Services",
      items: {
        type: "string"
      },
      uniqueItems: true
    },
    checklist: {
      type: "array",
      title: "Pre-Task Checklist",
      items: {
        type: "string"
      },
      uniqueItems: true
    },
    stops: {
      type: "array",
      title: "Additional Stops",
      items: {
        type: "object",
        properties: {
          location: {
            type: "string",
            title: "Stop Location",
            minLength: 3
          },
          sequence: {
            type: "number",
            title: "Sequence Number",
            minimum: 1
          },
          timing_details: {
            type: "string",
            title: "Timing Notes",
            maxLength: 200
          }
        },
        required: ["location", "sequence"]
      }
    }
  }
};

const CustomSelectWidget = (props) => {
  const { id, label, value, required, onChange, options, schema } = props;
  const [searchTerm, setSearchTerm] = useState("");
  const [items, setItems] = useState([]);
  const [loading, setLoading] = useState(false);
  
  // Use dropdown context to manage open state
  const { openDropdown, setOpenDropdown } = useContext(DropdownContext);
  const isOpen = openDropdown === id;

  const fetchItems = async () => {
    try {
      setLoading(true);
      let endpoint = '';
      
      if (schema.title === 'Additional Services') {
        endpoint = 'crud/addons?skip=0&limit=10&q=&status=';
      } else if (schema.title === 'Pre-Task Checklist') {
        endpoint = 'crud/checklist?skip=0&limit=10&q=&status=';
      }

      const response = await axios.get(endpoint);
      const fetchedItems = response.data?.items || [];
      
      setItems(fetchedItems.map(item => ({
        id: item.addons_id || item.checklist_id,
        name: item.name
      })));
    } catch (error) {
      console.error(`Error fetching ${schema.title}:`, error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (isOpen) {
      fetchItems();
    }
  }, [isOpen]);

  const filteredItems = items.filter(item =>
    item.name.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleSelect = (item) => {
    const currentValue = Array.isArray(value) ? value : [];
    const newValue = currentValue.includes(item.id) 
      ? currentValue.filter(v => v !== item.id)
      : [...currentValue, item.id];
    
    onChange(newValue);
  };

  const selectedItems = Array.isArray(value) 
    ? items.filter(item => value.includes(item.id))
    : [];

  const toggleDropdown = () => {
    if (isOpen) {
      setOpenDropdown(null);
    } else {
      setOpenDropdown(id);
    }
  };

  return (
    <div className="form-group" style={{ marginBottom: '20px', position: 'relative' }}>
      {/* <label htmlFor={id}>
        {label}
        {required && <span style={{ color: 'red' }}>*</span>}
      </label> */}
      
      <div 
        style={{ 
          border: '1px solid #ced4da',
          borderRadius: '4px',
          padding: '8px 12px',
          minHeight: '23px',
          cursor: 'pointer',
          backgroundColor: '#fff',
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'space-between',
          position: 'relative'
        }}
        onClick={toggleDropdown}
      >
        <div style={{ 
          display: 'flex', 
          flexWrap: 'wrap', 
          gap: '4px',
          flex: 1,
          overflow: 'hidden'
        }}>
          {selectedItems.length > 0 ? (
            selectedItems.map(item => (
              <span 
                key={item.id}
                style={{
                  backgroundColor: '#e9ecef',
                  borderRadius: '4px',
                  padding: '2px 6px',
                  fontSize: '14px',
                  display: 'inline-flex',
                  alignItems: 'center'
                }}
              >
                {item.name}
                <span 
                  style={{ 
                    marginLeft: '4px', 
                    cursor: 'pointer',
                    fontSize: '12px'
                  }}
                  onClick={(e) => {
                    e.stopPropagation();
                    handleSelect(item);
                  }}
                >
                  ×
                </span>
              </span>
            ))
          ) : (
            <span style={{ color: '#6c757d' }}>
              Select {label.toLowerCase()}
            </span>
          )}
        </div>
        <span style={{ 
          color: 'black', 
          marginLeft: '8px', 
          fontWeight: 'bold', 
          fontSize: '12px', 
          display: 'inline-block', 
          transform: 'scaleX(1.3)' 
        }}>
          {isOpen ? '˄' : '˅'}
        </span>
      </div>

      {isOpen && (
        <div 
          style={{
            position: 'absolute',
            zIndex: 1000,
            width: '100%',
            maxHeight: '300px',
            overflowY: 'auto',
            backgroundColor: 'white',
            border: '1px solid #ced4da',
            borderRadius: '4px',
            boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
            marginTop: '4px'
          }}
        >
          {loading && (
            <div style={{ padding: '8px', textAlign: 'center', color: '#6c757d' }}>
              Loading...
            </div>
          )}

          {!loading && filteredItems.length === 0 && (
            <div style={{ padding: '8px', textAlign: 'center', color: '#6c757d' }}>
              No {label.toLowerCase()} found
            </div>
          )}

          {!loading && filteredItems.length > 0 && (
            <ul style={{ listStyle: 'none', padding: 0, margin: 0 }}>
              {filteredItems.map(item => (
                <li 
                  key={item.id}
                  style={{
                    padding: '8px 12px',
                    cursor: 'pointer',
                    backgroundColor: value?.includes(item.id) ? '#f8f9fa' : 'white',
                    borderBottom: '1px solid #f8f9fa',
                    display: 'flex',
                    alignItems: 'center'
                  }}
                  onClick={(e) => {
                    e.stopPropagation();
                    handleSelect(item);
                  }}
                >
                  <input
                    type="radio"
                    checked={value?.includes(item.id)}
                    readOnly
                    style={{ 
                      marginRight: '8px',
                      accentColor: '#0d6efd'
                    }}
                  />
                  {item.name}
                </li>
              ))}
            </ul>
          )}
        </div>
      )}
    </div>
  );
};

const uiSchema = {
  "ui:order": [
    "customer_id",
    "pickup_location",
    "drop_location",
    "stops",
    "service",
    "task_mode",
    "schedule_id",
    "priority",
    "capacity",
    "roundtrip",
    "task_start_date_time",
    "task_end_date_time",
    "addons",
    "checklist"
  ],
  "ui:globalOptions": {
    styles: {
      input: {
        border: '1px solid #ced4da !important',
        borderRadius: '4px !important',
        padding: '8px 12px !important',
        fontSize: '14px !important',
        height: '38px !important'
      },
      select: {
        border: '1px solid #ced4da !important',
        borderRadius: '4px !important',
        padding: '8px 12px !important',
        fontSize: '14px !important',
        height: '38px !important'
      },
      textarea: {
        border: '1px solid #ced4da !important',
        borderRadius: '4px !important',
        padding: '8px 12px !important',
        fontSize: '14px !important'
      }
    }
  },
  customer_id: {
    "ui:placeholder": "Select a customer (optional)",
    "ui:widget": "select",
    "ui:options": {
      className: "uniform-field"
    }
  },
  pickup_location: {
    "ui:placeholder": "Search pickup location",
    "ui:options": {
      className: "uniform-field"
    },
    "ui:field": "locationField"
  },
  drop_location: {
    "ui:placeholder": "Search drop location",
    "ui:options": {
      className: "uniform-field"
    },
    "ui:field": "locationField"
  },
  service: {
    "ui:placeholder": "Select a service type",
    "ui:widget": "select",
    "ui:options": {
      className: "uniform-field"
    }
  },
  task_mode: {
    "ui:placeholder": "Select task mode",
    "ui:widget": "select",
    "ui:options": { className: "uniform-field" }
  },
  schedule_id: {
    "ui:placeholder": "Select a schedule",
    "ui:widget": "select",
    "ui:options": {
      className: "uniform-field"
    }
  },
  priority: {
    "ui:placeholder": "Select priority",
    "ui:widget": "select",
    "ui:options": { className: "uniform-field" }
  },
  capacity: {
    "ui:placeholder": "Enter required capacity (optional)",
    "ui:widget": "updown",
    "ui:options": { className: "uniform-field" }
  },
  roundtrip: {
    "ui:widget": "checkbox",
    "ui:options": {
      className: "uniform-checkbox",
      inline: true
    }
  },
  task_start_date_time: {
    "ui:widget": "datetime",
    "ui:options": { className: "uniform-field" }
  },
  task_end_date_time: {
    "ui:widget": "datetime",
    "ui:options": { className: "uniform-field" }
  },
  addons: {
    "ui:widget": CustomSelectWidget,
    "ui:options": {
      className: "uniform-field",
    }
  },
  checklist: {
    "ui:widget": CustomSelectWidget,
    "ui:options": {
      className: "uniform-field",
    }
  },
  stops: {
    "ui:options": {
      addable: true,
      orderable: true,
      removable: true,
      addButtonText: "+ Add Stop",
      deleteButtonText: "Remove",
      className: "uniform-array"
    },
    "ui:field": "stopsField",
    items: {
      "ui:order": ["location", "sequence", "timing_details"],
      location: {
        "ui:placeholder": "Search stop location",
        "ui:options": { className: "uniform-field" },
        "ui:field": "locationField"
      },
      sequence: {
        "ui:placeholder": "Enter sequence number",
        "ui:options": { className: "uniform-field" }
      },
      timing_details: {
        "ui:widget": "textarea",
        "ui:placeholder": "Enter timing notes",
        "ui:options": {
          className: "uniform-field",
          rows: 3
        }
      }
    }
  }
};

const LocationField = ({ formData, onChange, placeholder }) => {
  const inputRef = useRef(null);
  const [value, setValue] = useState(formData || '');

  useEffect(() => {
    if (window.google && window.google.maps && inputRef.current) {
      const autocomplete = new window.google.maps.places.Autocomplete(inputRef.current, {
        types: ['address']
      });
      autocomplete.addListener('place_changed', () => {
        const place = autocomplete.getPlace();
        if (place.formatted_address) {
          setValue(place.formatted_address);
          onChange(place.formatted_address);
        }
      });
    }
  }, []);

  return (
    <input
      type="text"
      className="uniform-field"
      value={value}
      onChange={(e) => {
        setValue(e.target.value);
        onChange(e.target.value);
      }}
      placeholder={placeholder}
      ref={inputRef}
      style={{
        border: '1px solid #ced4da',
        borderRadius: '4px',
        padding: '8px 12px',
        fontSize: '14px',
        width: '100%',
        outline: 'none',
        height: '38px'
      }}
    />
  );
};

const StopsField = (props) => {
  const { formData = [], onChange } = props;
  const [isHovered, setIsHovered] = useState(false);

  const handleAddStop = () => {
    onChange([...formData, { location: "", sequence: formData.length + 1 }]);
  };

  const handleRemoveStop = (index) => {
    const newStops = [...formData];
    newStops.splice(index, 1);
    onChange(newStops);
  };

  const updateStopField = (index, field, value) => {
    const newStops = [...formData];
    newStops[index] = { ...newStops[index], [field]: value };
    onChange(newStops);
  };

  return (
    <div className="stops-container" style={{ marginTop: '20px' }}>
      <div className="stops-header" style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
        <h4 style={{ margin: 0 }}>Additional Stops</h4>
        <button 
          type="button" 
          className="btn btn-add-stop"
          onClick={handleAddStop}
          style={{
            background: isHovered ? '#00D563': '#f0f0f0',
            border: '0px solid #f0f0f0',
            color: isHovered ? 'white' :'#00D563',
            cursor: 'pointer',
            padding: '6px 12px',
            borderRadius: '4px',
            fontSize: '14px',
            transition: 'all 0.3s ease'
          }}
          onMouseEnter={() => setIsHovered(true)}
          onMouseLeave={() => setIsHovered(false)}
        >
          + Add Stop
        </button>
      </div>
     
      {formData.map((stop, index) => (
        <div key={index} className="stop-item" style={{
          marginTop: '15px',
          padding: '15px',
          border: '1px solid #e9ecef',
          borderRadius: '4px',
          backgroundColor: '#f8f9fa'
        }}>
          <div className="stop-fields" style={{ display: 'grid', gap: '15px' }}>
            <div style={{ display: 'flex', flexDirection: 'column' }}>
              <label style={{ marginBottom: '5px', fontSize: '14px', fontWeight: '500' }}>Stop Location</label>
              <LocationField
                formData={stop.location}
                onChange={(value) => updateStopField(index, 'location', value)}
                placeholder="Search stop address"
              />
            </div>
           
            <div style={{ display: 'flex', flexDirection: 'column' }}>
              <label style={{ marginBottom: '5px', fontSize: '14px', fontWeight: '500' }}>Sequence Number</label>
              <input
                type="number"
                className="uniform-field"
                value={stop.sequence || index + 1}
                onChange={(e) => updateStopField(index, 'sequence', parseInt(e.target.value) || 1)}
                placeholder="Enter sequence number"
                min="1"
                style={{
                  border: '1px solid #ced4da',
                  borderRadius: '4px',
                  padding: '8px 12px',
                  fontSize: '14px',
                  outline: 'none',
                  height: '38px'
                }}
              />
            </div>
           
            <div style={{ display: 'flex', flexDirection: 'column' }}>
              <label style={{ marginBottom: '5px', fontSize: '14px', fontWeight: '500' }}>Timing Notes</label>
              <textarea
                className="uniform-field"
                value={stop.timing_details || ''}
                onChange={(e) => updateStopField(index, 'timing_details', e.target.value)}
                placeholder="Enter timing notes"
                rows="3"
                style={{
                  border: '1px solid #ced4da',
                  borderRadius: '4px',
                  padding: '8px 12px',
                  fontSize: '14px',
                  outline: 'none',
                  resize: 'vertical'
                }}
              />
            </div>
          </div>
         
          <button
            type="button"
            className="btn btn-remove-stop"
            onClick={() => handleRemoveStop(index)}
            style={{
              background: '#00D563',
              border: '1px solid white',
              color: 'white',
              cursor: 'pointer',
              padding: '6px 10px',
              borderRadius: '4px',
              fontSize: '14px',
              marginRight: '-18px',
              marginTop: '0'
            }}
          >
            Remove
          </button>
        </div>
      ))}
    </div>
  );
};

const TaskForm = ({ onSubmit, isSubmitting, initialData }) => {
  const [formData, setFormData] = useState({
    customer_id: '',
    service: '',
    task_mode: '',
    priority: '',
    roundtrip: false,
    capacity: 1,
    task_start_date_time: '',
    task_end_date_time: '',
    pickup_location: '',
    drop_location: '',
    addons: [],
    checklist: [],
    stops: [],
    ...initialData
  });

  const [services, setServices] = useState([]);
  const [addons, setAddons] = useState([]);
  const [checklists, setChecklists] = useState([]);
  const [customers, setCustomers] = useState([]);
  const [loading, setLoading] = useState(true);
  const [googleScriptLoaded, setGoogleScriptLoaded] = useState(false);
  const [formErrors, setFormErrors] = useState(null);
  const [buttonState, setButtonState] = useState('default');
  const [schedules, setSchedules] = useState([]);
  
  // State to track which dropdown is currently open
  const [openDropdown, setOpenDropdown] = useState(null);

  useEffect(() => {
    const fetchData = async () => {
      try {
        const accountId = localStorage.getItem('account_id');
        const [servicesRes, addonsRes, checklistsRes, schedulesRes, customersRes] = await Promise.all([
          axios.get('services/list/all/?skip=0&limit=10&q='),
          axios.get('crud/addons?skip=0&limit=10&q='),
          axios.get('crud/checklist?skip=0&limit=10&q='),
          axios.get(`schedules/?account_id=${accountId}`),
          axios.get(`customers/?account_id=${accountId}&skip=0&limit=10&q=&created_date_from=&created_date_to=&customer_type=&status=`)
        ]);

        setServices(
          servicesRes.data.users.map(service => ({
            value: service.service_id.toString(),
            label: service.service_title
          }))
        );

        setAddons(
          addonsRes.data.items.map(addon => ({
            value: addon.addons_id.toString(),
            label: addon.name
          }))
        );

        setChecklists(
          checklistsRes.data.items.map(item => ({
            value: item.checklist_id.toString(),
            label: item.name
          }))
        );

        setSchedules(
          schedulesRes.data.schedules.map(schedule => ({
            value: schedule.schedule_id,
            label: schedule.schedule_name
          }))
        );

        setCustomers(
          customersRes.data.users.map(customer => ({
            value: customer.id,
            label: `${customer.first_name} ${customer.last_name} (${customer.email})`
          }))
        );

        setLoading(false);
      } catch (error) {
        console.error("Error fetching form data:", error);
        setLoading(false);
      }
    };

    fetchData();
  }, []);

  const fields = {
    stopsField: StopsField,
    locationField: LocationField,
    CustomSelectWidget: CustomSelectWidget
  };

  const dynamicSchema = {
    ...taskSchema,
    properties: {
      ...taskSchema.properties,
      customer_id: {
        ...taskSchema.properties.customer_id,
        enum: customers.map(c => c.value),
        enumNames: customers.map(c => c.label)
      },
      service: {
        ...taskSchema.properties.service,
        enum: services.map(s => s.value),
        enumNames: services.map(s => s.label)
      },
      addons: {
        ...taskSchema.properties.addons,
      },
      checklist: {
        ...taskSchema.properties.checklist,
      },
      schedule_id: {
        type: "string",
        title: "Schedule",
        enum: schedules.map(s => s.value),
        enumNames: schedules.map(s => s.label)
      }
    }
  };

  const handleSubmit = ({ formData }, e) => {
    try {
      setButtonState('loading');
     
      if (!formData.service) {
        throw new Error("Please select a service type");
      }
      if (!formData.task_mode) {
        throw new Error("Please select a task mode");
      }
      if (!formData.priority) {
        throw new Error("Please select a priority level");
      }
      if (!formData.task_start_date_time || !formData.task_end_date_time) {
        throw new Error("Please provide both start and end dates");
      }

      const payload = {
        account_id: localStorage.getItem('account_id'),
        user_id: localStorage.getItem('user_id'),
        customer_id: formData.customer_id || null,
        service_id: String(formData.service),
        task_mode: formData.task_mode,
        priority: formData.priority,
        roundtrip: Boolean(formData.roundtrip),
        capacity: Number(formData.capacity) || 1,
        scheduled_start: formData.task_start_date_time,
        scheduled_end: formData.task_end_date_time,
        actual_start: null,
        actual_end: null,
        time_spent: 0,
        distance_travelled: 0.0,
        cost: 0.0,
        pickup_location: formData.pickup_location,
        drop_location: formData.drop_location,
        addons: Array.isArray(formData.addons) ? formData.addons : [],
        checklist: Array.isArray(formData.checklist) ? formData.checklist : [],
        stops: Array.isArray(formData.stops) ? formData.stops : [],
        status: "pending"
      };

      console.log("Submitting payload:", payload);
      onSubmit(payload);
      setFormErrors(null);
      setButtonState('success');
    } catch (error) {
      console.error("Form submission error:", error);
      setFormErrors(error.message);
      setButtonState('error');
      setTimeout(() => setButtonState('default'), 2000);
    }
  };

  if (loading) {
    return <div className="loading-indicator">Loading form options...</div>;
  }

  return (
    <DropdownContext.Provider value={{ openDropdown, setOpenDropdown }}>
      <div className="job-form-container">
        <style>
          {`
            ::-webkit-scrollbar {
              display: none;
            }
            .uniform-field {
              border: 1px solid #ced4da !important;
              border-radius: 4px !important;
              padding: 8px 12px !important;
              font-size: 14px !important;
              height: 38px !important;
              background: white !important;
              box-shadow: none !important;
              width: 100% !important;
              border: 1px solid #ced4da !important;
            }
           
            .uniform-field:focus {
              outline: none !important;
              border-color: #80bdff !important;
              box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25) !important;
            }
           
            .uniform-checkbox .form-check-input {
              border: 1px solid #ced4da !important;
              box-shadow: none !important;
              width: 18px !important;
              height: 18px !important;
              margin-right: 8px !important;
            }
           
            .uniform-array {
              border: none !important;
              padding: 0 !important;
            }
           
            .rjsf .form-group fieldset {
              border: none !important;
              padding: 0 !important;
              margin: 0 0 15px 0 !important;
            }
           
            .rjsf .panel {
              border: none !important;
              box-shadow: none !important;
              background: transparent !important;
            }
           
            .rjsf .panel-heading {
              border: none !important;
              background: transparent !important;
              padding: 10px 0 !important;
              font-size: 16px !important;
              font-weight: 500 !important;
            }
           
            .rjsf .panel-body {
              padding: 0 !important;
            }
           
            select.uniform-field {
              -webkit-appearance: none;
              -moz-appearance: none;
              appearance: none;
              background-image: url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='currentColor' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e");
              background-repeat: no-repeat;
              background-position: right 12px center;
              background-size: 16px;
            }
           
            .form-group label {
              font-weight: 500 !important;
              margin-bottom: 5px !important;
              display: block !important;
            }
           
            .array-item-move-up, .array-item-move-down {
              display: none !important;
            }
           
            .array-item-add button {
              background: #007bff !important;
              color: white !important;
              border: 1px solid #007bff !important;
              border-radius: 4px !important;
              padding: 6px 12px !important;
              font-size: 14px !important;
            }
           
            .array-item-remove {
              background: #dc3545 !important;
              color: white !important;
              border: 1px solid #dc3545 !important;
              border-radius: 4px !important;
              padding: 6px 12px !important;
              font-size: 14px !important;
            }
           
            .checkbox-inline {
              display: flex !important;
              align-items: center !important;
              margin-right: 15px !important;
            }
           
            .checkbox-inline input {
              margin-right: 8px !important;
            }
          `}
        </style>
       
        <Script
          url={`https://maps.googleapis.com/maps/api/js?key=AIzaSyDe-1PlmNj0J4kGAbZBJcelb_mlUof-6us&libraries=places`}
          onLoad={() => setGoogleScriptLoaded(true)}
          onError={() => console.error("Failed to load Google Maps API")}
        />
       
        {!googleScriptLoaded && (
          <div className="loading-indicator">Loading Google Maps...</div>
        )}
       
        {formErrors && (
          <div className="alert alert-danger" style={{
            padding: '10px 15px',
            border: '1px solid #f5c6cb',
            borderRadius: '4px',
            backgroundColor: '#f8d7da',
            color: '#721c24',
            marginBottom: '20px'
          }}>
            {formErrors}
          </div>
        )}
       
        {googleScriptLoaded && (
          <Form
            schema={dynamicSchema}
            uiSchema={uiSchema}
            formData={formData}
            onChange={({ formData }) => setFormData(formData)}
            onSubmit={handleSubmit}
            validator={validator}
            showErrorList={false}
            disabled={isSubmitting || buttonState === 'loading'}
            fields={fields}
            noHtml5Validate
          >
            <div className="modal-buttons" style={{
              marginTop: '20px',
              display: 'flex',
              gap: '10px',
              justifyContent: 'flex-start'
            }}>
              <button
                type="submit"
                className={`btn ${buttonState === 'success' ? 'btn-success' : buttonState === 'error' ? 'btn-danger' : 'btn-primary'}`}
                disabled={isSubmitting || buttonState === 'loading'}
                style={{
                  padding: '8px 16px',
                  borderRadius: '4px',
                  cursor: 'pointer',
                  fontSize: '14px',
                  fontWeight: 'bold',
                  backgroundColor: buttonState === 'success' ? '#28a745' :
                                  buttonState === 'error' ? '#dc3545' :
                                  '#f0f0f0',
                  color: buttonState === 'success' ? 'white' :
                        buttonState === 'error' ? 'white' :
                        '#00D563',
                  border: '1px solid white',
                  transition: 'all 0.3s ease'
                }}
                onMouseEnter={(e) => {
                  if (buttonState === 'default') {
                    e.currentTarget.style.backgroundColor = '#00D563';
                    e.currentTarget.style.color = '#f0f0f0';
                  }
                }}
                onMouseLeave={(e) => {
                  if (buttonState === 'default') {
                    e.currentTarget.style.backgroundColor = '#f0f0f0';
                    e.currentTarget.style.color = '#00D563';
                  }
                }}
              >
                {buttonState === 'loading' ? (
                  <>
                    <span
                      className="spinner-border spinner-border-sm"
                      role="status"
                      aria-hidden="true"
                      style={{
                        marginRight: '5px',
                        verticalAlign: 'middle',
                        borderColor: '#f0f0f0'
                      }}
                    ></span>
                    Submitting...
                  </>
                ) :
                buttonState === 'success' ? 'Success!' :
                buttonState === 'error' ? 'Error!' : 'Submit'}
              </button>
            </div>
          </Form>
        )}
      </div>
    </DropdownContext.Provider>
  );
};

export default TaskForm;