import React, { useState } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import TaskForm from './TaskForm';

const AddTask = ({ onClose, onCreateSuccess }) => {
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [error, setError] = useState(null);

    const transformStops = (formData) => {
        const task_stops = [];
        let sequence = 1;

        if (formData.pickup_location) {
            task_stops.push({
                location: formData.pickup_location,
                type: "pickup",
                sequence: sequence++,
                timing: null
            });
        }

        if (formData.stops?.length) {
            formData.stops.forEach(stop => {
                task_stops.push({
                    location: stop.location,
                    type: "intermediate",
                    sequence: sequence++,
                    timing: stop.timing_details || null
                });
            });
        }

        if (formData.drop_location) {
            task_stops.push({
                location: formData.drop_location,
                type: "dropoff",
                sequence: sequence,
                timing: null
            });
        }

        return task_stops;
    };

    const handleSubmit = async (formData) => {
        setIsSubmitting(true);
        setError(null);
        
        try {
            const account_id = localStorage.getItem('account_id');
            const customer_id = localStorage.getItem('user_id');
    
            if (!account_id || !customer_id) {
                throw new Error('User account information not found');
            }
    
            const addonsDict = formData.addons?.reduce((acc, id) => {
                acc[id] = true;
                return acc;
            }, {}) || {};
    
            const checklistDict = formData.checklist?.reduce((acc, id) => {
                acc[id] = true;
                return acc;
            }, {}) || {};
    
            const payload = {
                account_id: account_id,
                service_id: formData.service_id || formData.service,
                schedule_id: null,
                customer_id: customer_id,
                task_mode: formData.task_mode || "scheduled",
                priority: formData.priority || "normal",
                roundtrip: formData.roundtrip || false,
                capacity: formData.capacity || 1,
                scheduled_start: formData.scheduled_start || formData.task_start_date_time,
                scheduled_end: formData.scheduled_end || formData.task_end_date_time,
                status: "pending",
                addons: addonsDict,
                checklist: checklistDict,
                variables: {},
                task_stops: transformStops(formData),
                actual_start: null,
                actual_end: null,
                time_spent: 0,
                distance_travelled: 0.0,
                cost: 0.0
            };
    
            const res = await axios.post('/tasks/', payload);
            onCreateSuccess?.(res.data);
            onClose?.();
            return true;
        } catch (err) {
            setError(err.response?.data?.message || err.message || "Error creating task. Please try again.");
            console.error("Task creation error:", err);
            return false;
        } finally {
            setIsSubmitting(false);
        }
    };

    return (
        <div style={{ maxWidth: '600px', padding: '20px' }}>
            {error && <div className="error-message">{error}</div>}
            <TaskForm 
                onSubmit={handleSubmit} 
                isSubmitting={isSubmitting}
            />
        </div>
    );
};

export default AddTask;