import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import '../../themes/blue/Workforceview.css';

const TAB_CONFIG = [
  { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
  //{ id: 'details', label: 'Details', icon: 'info'},
  //{ id: 'activity', label: 'Activity', icon: 'history' }
];

const IoTDeviceView = ({ deviceId, deviceData: initialData, onClose }) => {
  const [deviceData, setDeviceData] = useState(initialData || null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');
  const [hoveredTab, setHoveredTab] = useState(null);

  useEffect(() => {
    if (!initialData) {
      axios.get(`/iotdevices/${deviceId}`)
        .then(res => setDeviceData(res.data))
        .catch(() => setError('Error loading device details'))
        .finally(() => setLoading(false));
    } else {
      setLoading(false);
    }
  }, [deviceId]);

  const renderTabContent = () => {
    if (!deviceData) return null;

    switch (activeTab) {
   case 'dashboard':
  return (
    <div className="tab-content">
      <h4>
        <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>
          dashboard
        </span>
        Device Summary
      </h4>
      <p><strong>Device ID:</strong> {deviceData.device_id || deviceData.id}</p>
      <p><strong>Type:</strong> {deviceData.device_type}</p>
      <p><strong>Status:</strong> {deviceData.status}</p>
      <p><strong>Associated Entity:</strong> {deviceData.associated_entity_type} (ID: {deviceData.associated_entity_id})</p>
      <p><strong>Installed On:</strong> {new Date(deviceData.created_date).toLocaleDateString()}</p>
      <p><strong>Installation Details:</strong> {deviceData.installation_details || 'N/A'}</p>
      <p><strong>Account ID:</strong> {deviceData.account_id}</p>
      <p><strong>User ID:</strong> {deviceData.user_id || 'N/A'}</p>
      <p><strong>Last Updated:</strong> {new Date(deviceData.last_updated).toLocaleString()}</p>
    </div>
  );


      case 'details':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>info</span>
              Device Details
            </h4>
            <div className="detail-grid">
              <div className="detail-row"><strong>Device Type:</strong> {deviceData.device_type}</div>
              <div className="detail-row"><strong>Associated Entity:</strong> {deviceData.associated_entity_type}</div>
              <div className="detail-row"><strong>Entity ID:</strong> {deviceData.associated_entity_id}</div>
              <div className="detail-row"><strong>Installation Details:</strong> {deviceData.installation_details || '--'}</div>
              <div className="detail-row"><strong>Status:</strong> {deviceData.status}</div>
              <div className="detail-row"><strong>Last Updated:</strong> {new Date(deviceData.last_updated).toLocaleString()}</div>
              <div className="detail-row"><strong>Created Date:</strong> {new Date(deviceData.created_date).toLocaleString()}</div>
            </div>
          </div>
        );

      case 'activity':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>history</span>
              Recent Activity
            </h4>
            <p>Activity logs would be displayed here</p>
          </div>
        );

      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading device data...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }

  return (
    <Modal onClose={onClose}>
      <div className="fixed top-0 right-0 w-full md:w-[55%] lg:w-[45%] xl:w-[40%] h-full bg-white shadow-lg z-50 overflow-hidden">
        {/* Header Section */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          padding: '10px 10px 20px',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px'
        }}>
          <div style={{ display: 'flex', alignItems: 'center' }}>
            <div style={{
              width: '60px',
              height: '60px',
              borderRadius: '50%',
              backgroundColor: '#e0e0e0',
              color: '#000',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              fontSize: '20px',
              fontWeight: 'bold',
              marginRight: '16px'
            }}>
              {deviceData.device_type?.charAt(0).toUpperCase()}
            </div>
            <div>
              <div style={{
                fontWeight: '600',
                fontSize: '20px',
                lineHeight: '1.4',
                color: 'black'
              }}>
                {deviceData.device_type} Device
              </div>
              <div style={{
                fontSize: '16px',
                color: 'black',
                marginTop: '10px',
                fontWeight: 'bold'
              }}>
                ID: {deviceData.device_id}
              </div>
            </div>
          </div>
        </div>

        {/* Tabs Section */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '0px 10px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '20px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px',
                transition: 'min-width 0.2s ease'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && (
                <span>{tab.label}</span>
              )}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-1px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Tab Content */}
        <div style={{ maxHeight: 'calc(100vh - 250px)', overflowY: 'auto', padding: '0 20px' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default IoTDeviceView;