import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import IoTDeviceView from './IoTDeviceView';
import IoTDeviceUpdate from './IoTDeviceUpdate';
import IoTDeviceDelete from './IoTDeviceDelete';


const IOT_DEVICE_HEADER_TO_KEY_MAP = {
  "Device Type": {
    key: "device_type",
    label: "Device Type",
    sortable: true,
    clickable: true
  },
  "Entity Type": {
    key: "associated_entity_type",
    label: "Entity Type",
    sortable: true
  },
  "Model": {
    key: "model_name",  // Updated from "model" to "model_name"
    label: "Model",
    sortable: true
  },
  "UUID": {
    key: "uuid",
    label: "UUID",
    sortable: true
  },
  "Installation Details": {
    key: "installation_details",
    label: "Installation Details",
    sortable: true
  },
  "Status": {
    key: "status",
    label: "Status",
    sortable: true
  },
  "Installed On": {
    key: "created_date",
    label: "Installed On",
    sortable: false
  }
};


const IoTDevicesList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const [selectedDevice, setSelectedDevice] = useState(null);
  const [selectedDeviceFromCell, setSelectedDeviceFromCell] = useState(null);
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  const account_id = localStorage.getItem('account_id');
  const fetchDataUrl = `/iotdevices/?account_id=${account_id}`;


  const actionFormats = [
    { label: "view", modal: true, url: "/iotdevices/view/{id}" },
    { label: "edit", modal: true, url: "/iotdevices/update/{id}" },
    { label: "delete", modal: true, url: "/iotdevices/delete/{id}" },
  ];


  const searchFields = [
        {
      name: 'date_from',
      type: 'date',
      placeholder: 'From Date'
    },
    {
      name: 'date_to',
      type: 'date',
      placeholder: 'To Date'
    },
        {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All Statuses', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' },
        { label: 'Faulty', value: 'faulty' }
      ]
    },
    {
      name: 'q',
      type: 'text',
      placeholder: 'Search by device ID, type...',
      defaultValue: initialSearchQuery
    },
    {
      name: 'device_type',
      type: 'dropdown',
      placeholder: 'Device Type',
      options: [
        { label: 'All Types', value: '' },
        { label: 'OBDII', value: 'obdii' },
        { label: 'GPS', value: 'gps' },
        { label: 'Mobile', value: 'mobile' },
        { label: 'Dashcam', value: 'dashcam' },
        { label: 'Seat Sensor', value: 'seat_sensor' },
        { label: 'Heat Sensor', value: 'heat_sensor' },
        { label: 'Fuel Sensor', value: 'fuel_sensor' },
        { label: 'Camera', value: 'camera' },
        { label: 'Proximity Sensor', value: 'proximity_sensor' },
        { label: 'Others', value: 'others' }
      ]
    },
  ];


  const handleActionClick = (deviceId, action) => {
    if (action.label === "view") {
      setSelectedDevice({ deviceId, modalType: "view" });
    } else if (action.label === "edit") {
      setSelectedDevice({ deviceId, modalType: "update" });
    } else if (action.label === "delete") {
      setSelectedDevice({ deviceId, modalType: "delete" });
    }
  };


  const handleCellClick = (header, item) => {
    if (header === "Device ID") {
      setSelectedDeviceFromCell({
        deviceId: item.device_id,
        modalType: "view"
      });
    }
  };


  const closeModal = () => {
    setSelectedDevice(null);
    setSelectedDeviceFromCell(null);
  };


  const refreshList = () => setRefreshTrigger(prev => prev + 1);


  return (
    <div>
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={IOT_DEVICE_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}
        refreshTrigger={refreshTrigger}
        searchFields={searchFields}
      />


      {selectedDevice?.modalType === "view" && (
        <IoTDeviceView
          deviceId={selectedDevice.deviceId}
          onClose={closeModal}
        />
      )}


      {selectedDevice?.modalType === "update" && (
        <IoTDeviceUpdate
          deviceId={selectedDevice.deviceId}
          onClose={closeModal}
          onUpdateSuccess={refreshList}
        />
      )}


      {selectedDevice?.modalType === "delete" && (
        <IoTDeviceDelete
          deviceId={selectedDevice.deviceId}
          onClose={closeModal}
          onDeleteSuccess={refreshList}
        />
      )}


      {selectedDeviceFromCell?.modalType === "view" && (
        <IoTDeviceView
          deviceId={selectedDeviceFromCell.deviceId}
          onClose={closeModal}
        />
      )}
    </div>
  );
};


export default IoTDevicesList;