import React, { useState, useEffect } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';


export const iotDeviceSchema = {
  type: "object",
  required: [
    "device_type",
    "associated_entity_type",
    "associated_entity_id",
    "status"
  ],
  properties: {
    device_type: {
      type: "string",
      title: "Device Type",
      enum: [
        "obdii",
        "gps",
        "mobile",
        "dashcam",
        "seat_sensor",
        "heat_sensor",
        "others"
      ]
    },
    associated_entity_type: {
      type: "string",
      title: "Associated Entity Type",
      enum: ["vehicle", "workforce"]
    },
    associated_entity_id: {
      type: "string",
      title: "Associated Entity ID",
      enum: [],
      enumNames: []
    },
    model: {
      type: "string",
      title: "Device Model (Optional)"
    },
    uuid: {
      type: "string",
      title: "Device UUID (Optional)"
    },
    installation_details: {
      type: "string",
      title: "Installation Details",
      default: ""
    },
    // status: {
    //   type: "string",
    //   title: "Status",
    //   enum: ["active", "inactive", "faulty"],
    //   default: "active"
    // }
  }
};


export const iotDeviceUiSchema = {
  "ui:order": [
    "device_type",
    "associated_entity_type",
    "associated_entity_id",
    "model",
    "uuid",
    "installation_details",
    "status"
  ],
  "ui:submitButtonOptions": {
    norender: false,
    submitText: "Submit"
  },
  associated_entity_type: {
    "ui:widget": "radio",
    "ui:options": {
      inline: true
    }
  },
  associated_entity_id: {
    "ui:help": "Select from the list based on entity type",
    "ui:options": {
      updateSchema: true
    }
  },
  installation_details: {
    "ui:widget": "textarea",
    "ui:options": {
      rows: 3
    }
  },
  // status: {
  //   "ui:widget": "select"
  // }
};


const IoTDeviceForm = ({ initialData = {}, onSubmit }) => {
  const [formData, setFormData] = useState({
    status: "active",
    account_id: localStorage.getItem('account_id'),
    user_id: localStorage.getItem('user_id'),
    ...initialData
  });


  const [schema, setSchema] = useState(iotDeviceSchema);
  const [submitted, setSubmitted] = useState(false);
  const [loadingEntities, setLoadingEntities] = useState(false);


  useEffect(() => {
    const fetchEntities = async () => {
      if (!formData.associated_entity_type) return;


      setLoadingEntities(true);
      try {
        let url, dataKey, idKey, nameKey;


        if (formData.associated_entity_type === 'workforce') {
          url = 'workforce/list/?skip=0&limit=100';
          dataKey = 'workforce';
          idKey = 'workforce_id';
          nameKey = 'first_name';
        } else {
          url = 'fleets/list/all/?skip=0&limit=100';
          dataKey = 'fleets';
          idKey = 'fleet_id';
          nameKey = 'fleet_name';
        }


        const response = await axios.get(url);
        const entities = response.data[dataKey] || [];


        const entityOptions = entities.map(entity => ({
          value: entity[idKey],
          label: entity[nameKey] + (entity.last_name ? ` ${entity.last_name}` : '')
        }));


        setSchema(prevSchema => ({
          ...prevSchema,
          properties: {
            ...prevSchema.properties,
            associated_entity_id: {
              ...prevSchema.properties.associated_entity_id,
              enum: entityOptions.map(opt => opt.value),
              enumNames: entityOptions.map(opt => opt.label)
            }
          }
        }));


        if (
          formData.associated_entity_id &&
          !entityOptions.some(opt => opt.value === formData.associated_entity_id)
        ) {
          setFormData(prev => ({ ...prev, associated_entity_id: '' }));
        }


      } catch (error) {
        console.error('Error fetching entities:', error);
      } finally {
        setLoadingEntities(false);
      }
    };


    fetchEntities();
  }, [formData.associated_entity_type]);


  const handleSubmit = async ({ formData }) => {
    setSubmitted(true);


    const transformedData = {
      ...formData,
      model_name: formData.model || null,
      uuid: formData.uuid || null
    };
    delete transformedData.model;


    onSubmit(transformedData);
  };


  return (
    <div style={{ maxWidth: "600px", padding: "20px" }}>
      <Form
        schema={schema}
        uiSchema={iotDeviceUiSchema}
        formData={formData}
        onChange={({ formData }) => setFormData(formData)}
        onSubmit={handleSubmit}
        validator={validator}
        showErrorList={submitted}
        noHtml5Validate
      />
      {loadingEntities && <p>Loading {formData.associated_entity_type} list...</p>}
    </div>
  );
};


export default IoTDeviceForm;