import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';

const TAB_CONFIG = [
  { id: 'details', label: 'Details', icon: 'info' },
  { id: 'geofence', label: 'Geofence', icon: 'fence' },
  { id: 'rule', label: 'Rule', icon: 'rule' }
];

const GeofencingRuleMappingView = ({ mappingId, onClose }) => {
  const [mappingData, setMappingData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('details');
  const [hoveredTab, setHoveredTab] = useState(null);

  useEffect(() => {
    axios.get(`/geofence_rule_mapping/${mappingId}`)
      .then(res => {
        setMappingData(res.data);
        setLoading(false);
      })
      .catch(err => {
        console.error("Error fetching mapping:", err);
        setError('Error fetching mapping details.');
        setLoading(false);
      });
  }, [mappingId]);

  const renderTabContent = () => {
    if (!mappingData) return null;

    switch (activeTab) {
      case 'details':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '12px',
                color: '#333'
              }}>
                info
              </span>
              Mapping Details
            </h4>

            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <p><strong>Status:</strong> {mappingData.status || 'Active'}</p>
              <p><strong>Created:</strong> {new Date(mappingData.created_date).toLocaleDateString()}</p>
              <p><strong>Last Updated:</strong> {new Date(mappingData.updated_date).toLocaleDateString()}</p>
              <p><strong>Assigned Entity Type:</strong> {mappingData.assigned_entity_type || '—'}</p>
            </div>
          </div>
        );

      case 'geofence':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '12px',
                color: '#333'
              }}>
                fence
              </span>
              Geofence Details
            </h4>

            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <p><strong>Name:</strong> {mappingData.geofence_name || '—'}</p>
              <p><strong>ID:</strong> {mappingData.geofence_id || '—'}</p>
              <p><strong>Shape:</strong> {mappingData.geofence_shape || '—'}</p>
            </div>
          </div>
        );

      case 'rule':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '12px',
                color: '#333'
              }}>
                rule
              </span>
              Rule Details
            </h4>

            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <p><strong>Name:</strong> {mappingData.rule_name || '—'}</p>
              <p><strong>ID:</strong> {mappingData.rule_id || '—'}</p>
              <p><strong>Description:</strong> {mappingData.rule_description || '—'}</p>
              <p><strong>Type:</strong> {mappingData.rule_type || '—'}</p>
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading mapping data...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }

  return (
    <Modal onClose={onClose}>
      <div style={{ height: '100%', overflow: 'hidden' }}>
        {/* Header with first letter avatar - FOR MAPPING COMPONENT */}
<div style={{
  display: 'flex',
  alignItems: 'center',
  gap: '20px',
  padding: '0px 10px 10px',
  borderBottom: '1px solid #e0e0e0'
}}>
  <div style={{
    width: '60px',
    height: '60px',
    borderRadius: '50%',
    backgroundColor: '#e0e0e0',
    color: '#000',
    display: 'flex',
    alignItems: 'center',
    justifyContent: 'center',
    fontSize: '24px',
    fontWeight: 'bold',
    textTransform: 'uppercase'
  }}>
    {mappingData?.name?.charAt(0) || 'G'}  {/* Fallback to 'G' */}
  </div>
  <div>
    <h3 style={{ margin: '0px', fontWeight: 'bold', fontSize: '20px', color: 'black' }}>
      {mappingData?.name || 'Geofence'}  {/* Fallback to 'Geofence' */}
    </h3>
    <p style={{ marginTop: '5px', fontSize: '16px', color: 'black', fontWeight: 'bold' }}>
      ID: #{mappingData?.id}
    </p>
  </div>
</div>


        {/* Tabs */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '20px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && <span>{tab.label}</span>}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-2px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Content */}
        <div style={{ padding: '0 20px', overflow: 'auto', maxHeight: '400px' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default GeofencingRuleMappingView;