import React, { useState, useEffect } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import GeofencingRuleMappingView from './GeofencingRuleMappingView';
import GeofencingRuleMappingUpdate from './GeofencingRuleMappingUpdate';
import GeofencingRuleMappingDelete from './GeofencingRuleMappingDelete';


const MAPPING_HEADER_TO_KEY_MAP = {
  "Geofence": {
    key: "geofence_name",
    label: "Geofence",
    sortable: true,
    clickable: true
  },
  "Rule": {
    key: "rule_name",
    label: "Rule",
    sortable: true
  },
  "Vehicle Type": {
    key: "vehicle_types",
    label: "Vehicle Type",
    sortable: false,
    format: (value) => Array.isArray(value) ? value.join(', ') : ''
  },
  "Workforce Type": {
    key: "workforce_types",
    label: "Workforce Type",
    sortable: false,
    format: (value) => Array.isArray(value) ? value.join(', ') : ''
  },
  "Created Date": {
    key: "created_date",
    label: "Created Date",
    sortable: true
  }
};


const GeofencingRuleMappingList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const [accountId, setAccountId] = useState(null);
  const [userId, setUserId] = useState(null);
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery
  });
  const [selectedMapping, setSelectedMapping] = useState(null);
 
  useEffect(() => {
    const storedAccountId = localStorage.getItem('account_id');
    const storedUserId = localStorage.getItem('user_id');
    if (storedAccountId) setAccountId(storedAccountId);
    if (storedUserId) setUserId(storedUserId);
  }, []);


  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search by geofence or rule...', defaultValue: initialSearchQuery },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' }
      ]
    }
  ];


  const actionFormats = [
    { label: "view", modal: true, url: "/geofence_rule_mapping/{id}" },
    { label: "edit", modal: true, url: "/geofence_rule_mapping/{id}" },
    { label: "delete", modal: true, url: "/geofence_rule_mapping/{id}" }
  ];


  const handleActionClick = (mappingId, action) => {
    setSelectedMapping({ mappingId, modalType: action.label });
  };


  const buildFetchUrl = () => {
    if (!accountId) return '';
    let url = `/geofence_rule_mapping/?account_id=${accountId}`;
    if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
    if (searchParams.status) url += `&status=${searchParams.status}`;
    return url;
  };


  if (!accountId || !userId) return <div>Loading...</div>;


  return (
    <div>
      <SearchPage
        fetchDataUrl={buildFetchUrl()}
        ListComponent={List}
        itemKeyMapping={MAPPING_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="id"
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        searchFields={searchFields}
      />


      {selectedMapping?.modalType === "view" && (
        <GeofencingRuleMappingView
          mappingId={selectedMapping.mappingId}
          onClose={() => setSelectedMapping(null)}
        />
      )}
      {selectedMapping?.modalType === "edit" && (
        <GeofencingRuleMappingUpdate
          mappingId={selectedMapping.mappingId}
          onClose={() => setSelectedMapping(null)}
          onUpdateSuccess={() => setSelectedMapping(null)}
        />
      )}
      {selectedMapping?.modalType === "delete" && (
        <GeofencingRuleMappingDelete
          mappingId={selectedMapping.mappingId}
          onClose={() => setSelectedMapping(null)}
          onDeleteSuccess={() => setSelectedMapping(null)}
        />
      )}
    </div>
  );
};


export default GeofencingRuleMappingList;