import React, { useState, useEffect } from 'react';
import axios from 'axios';
import {
  Typography,
  Select,
  MenuItem,
  FormControl,
  TextField,
  Autocomplete,
  Box,
  Divider
} from '@mui/material';

const GeofencingRuleMappingForm = ({ initialData = {}, onSubmit }) => {
  const [formData, setFormData] = useState({
    account_id: localStorage.getItem('account_id'),
    user_id: localStorage.getItem('user_id'),
    status: "active",
    geofence_id: initialData.geofence_id || '',
    rule_id: initialData.rule_id || '',
  });

  const [geofences, setGeofences] = useState([]);
  const [rules, setRules] = useState([]);
  const [loading, setLoading] = useState({ geofences: false, rules: false });

  const [vehicleSearchQuery, setVehicleSearchQuery] = useState('');
  const [vehicleSearchResults, setVehicleSearchResults] = useState([]);
  const [vehicleSearchLoading, setVehicleSearchLoading] = useState(false);
  const [selectedVehicles, setSelectedVehicles] = useState([]);
  const [vehicleDropdownOpen, setVehicleDropdownOpen] = useState(false);

  const [workforceSearchQuery, setWorkforceSearchQuery] = useState('');
  const [workforceSearchResults, setWorkforceSearchResults] = useState([]);
  const [workforceSearchLoading, setWorkforceSearchLoading] = useState(false);
  const [selectedWorkforce, setSelectedWorkforce] = useState([]);
  const [workforceDropdownOpen, setWorkforceDropdownOpen] = useState(false);

  useEffect(() => {
    const fetchStaticData = async () => {
      try {
        setLoading({ geofences: true, rules: true });
        const [geofencesRes, rulesRes] = await Promise.all([
          axios.get('/geofences/'),
          axios.get('/geofencerules/')
        ]);

        setGeofences(geofencesRes.data?.users?.map(g => ({ id: g.id, name: g.name })) || []);
        setRules(rulesRes.data?.users?.map(r => ({ id: r.id, name: r.rule_name })) || []);
      } catch (error) {
        console.error('Error fetching geofences/rules:', error);
      } finally {
        setLoading({ geofences: false, rules: false });
      }
    };

    fetchStaticData();
  }, []);

  useEffect(() => {
    if (initialData.assigned_entities) {
      const vehicles = initialData.assigned_entities.filter(e => e.entity_type === 'vehicle')
        .map(e => ({ id: e.id, name: '' }));
      const workforce = initialData.assigned_entities.filter(e => e.entity_type === 'workforce')
        .map(e => ({ id: e.id, name: '' }));
      setSelectedVehicles(vehicles);
      setSelectedWorkforce(workforce);
    }
  }, [initialData]);

  useEffect(() => {
    const delayDebounce = setTimeout(() => {
      if (vehicleSearchQuery.trim().length >= 3) {
        fetchVehicles(vehicleSearchQuery.trim());
        setVehicleDropdownOpen(true);
      } else {
        setVehicleDropdownOpen(false);
      }
    }, 300);
    return () => clearTimeout(delayDebounce);
  }, [vehicleSearchQuery]);

  useEffect(() => {
    const delayDebounce = setTimeout(() => {
      if (workforceSearchQuery.trim().length >= 3) {
        fetchWorkforce(workforceSearchQuery.trim());
        setWorkforceDropdownOpen(true);
      } else {
        setWorkforceDropdownOpen(false);
      }
    }, 300);
    return () => clearTimeout(delayDebounce);
  }, [workforceSearchQuery]);

  const fetchVehicles = async (query) => {
    setVehicleSearchLoading(true);
    try {
      const token = localStorage.getItem('token');
      const res = await axios.get(
        `fleets/list/all/?skip=0&limit=10&q=${encodeURIComponent(query)}`,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      const vehicles = res.data?.fleets?.map(v => ({ id: v.fleet_id, name: v.fleet_name })) || [];

      setVehicleSearchResults(vehicles);
      setSelectedVehicles(prev =>
        prev.map(vehicle => {
          const found = vehicles.find(v => v.id === vehicle.id);
          return found ? { ...vehicle, name: found.name } : vehicle;
        })
      );
    } catch (err) {
      console.error('Error fetching vehicles:', err);
      setVehicleSearchResults([]);
    } finally {
      setVehicleSearchLoading(false);
    }
  };

  const fetchWorkforce = async (query) => {
    setWorkforceSearchLoading(true);
    try {
      const token = localStorage.getItem('token');
      const res = await axios.get(
        `workforce/dropdown/workforce?q=${encodeURIComponent(query)}`,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      const workforce = res.data?.workforce?.map(w => ({
        id: w.workforce_id,
        name: `${w.first_name} ${w.last_name || ''}`.trim()
      })) || [];

      setWorkforceSearchResults(workforce);
      setSelectedWorkforce(prev =>
        prev.map(worker => {
          const found = workforce.find(w => w.id === worker.id);
          return found ? { ...worker, name: found.name } : worker;
        })
      );
    } catch (err) {
      console.error('Error fetching workforce:', err);
      setWorkforceSearchResults([]);
    } finally {
      setWorkforceSearchLoading(false);
    }
  };

  const handleSubmit = (e) => {
    e.preventDefault();
    const payload = {
      geofence_id: formData.geofence_id,
      geofence_name: geofences.find(g => g.id === formData.geofence_id)?.name || '',
      rule_id: formData.rule_id,
      rule_name: rules.find(r => r.id === formData.rule_id)?.name || '',
      assigned_entities: [
        ...selectedVehicles.map(v => ({ id: v.id, entity_type: 'vehicle' })),
        ...selectedWorkforce.map(w => ({ id: w.id, entity_type: 'workforce' }))
      ],
      account_id: formData.account_id,
      user_id: formData.user_id,
      status: formData.status
    };
    onSubmit(payload);
  };

  // Add Workforce-style styles
  useEffect(() => {
    const style = document.createElement("style");
    style.innerHTML = `
      .boxed-input input,
      .boxed-input select,
      .boxed-input textarea {
        border: 1px solid #ccc;
        border-radius: 4px;
        padding: 10px; /* Reduced for geofence/rule */
        background-color: #fff;
        box-shadow: none;
        width: 100%;
        font-size: 14px;
      }

      .boxed-input input:focus,
      .boxed-input select:focus,
      .boxed-input textarea:focus {
        border-color: #000;
        outline: none;
      }

      .boxed-input label {
        font-weight: 500;
        margin-bottom: 6px;
        display: block;
        text-align: left;
      }

      /* Specific adjustments for MuiOutlinedInput-root, ensuring overrides where needed */
      .MuiOutlinedInput-root {
        background-color: #fff !important; 
        border-radius: 4px;
        padding: 4px 10px; /* Base padding for MuiOutlinedInput-root */
        font-size: 14px;
        border: 1px solid #ccc; 
      }

      .MuiOutlinedInput-notchedOutline {
        border: none !important;
      }

      button[type="submit"] {
        background-color: #00c853;
        color: white;
        padding: 8px 16px;
        font-weight: bold;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        margin-top: 16px;
      }

      button[type="submit"]:hover {
        background-color: #009624;
      }
    `;
    document.head.appendChild(style);
    return () => document.head.removeChild(style);
  }, []);

  return (
    <div style={{ padding: '24px 24px 24px 0', maxWidth: 600 }}>
      <Box component="form" onSubmit={handleSubmit} className="boxed-input">
        {/* Geofence Dropdown */}
        <FormControl fullWidth sx={{ mb: 3 }}>
          <label>Geofence</label>
          <Select
            value={formData.geofence_id}
            onChange={(e) => setFormData({ ...formData, geofence_id: e.target.value })}
            required
            size="small" // Keeps it compact
          >
            {geofences.map(g => (
              <MenuItem key={g.id} value={g.id}>{g.name}</MenuItem>
            ))}
          </Select>
        </FormControl>

        {/* Rule Dropdown */}
        <FormControl fullWidth sx={{ mb: 3 }}>
          <label>Rule</label>
          <Select
            value={formData.rule_id}
            onChange={(e) => setFormData({ ...formData, rule_id: e.target.value })}
            required
            size="small" // Keeps it compact
          >
            {rules.map(r => (
              <MenuItem key={r.id} value={r.id}>{r.name}</MenuItem>
            ))}
          </Select>
        </FormControl>

        {/* Vehicles Autocomplete */}
        <FormControl fullWidth  >
          <label>Vehicles</label>
          <Autocomplete
            multiple
            options={vehicleSearchResults}
            getOptionLabel={(option) => option.name || ''}
            inputValue={vehicleSearchQuery}
            onInputChange={(_, newValue, reason) => {
              if (reason === 'input') setVehicleSearchQuery(newValue);
            }}
            onChange={(_, newValue) => {
              setSelectedVehicles(newValue);
              setVehicleSearchQuery('');
            }}
            filterOptions={(x) => x}
            open={vehicleDropdownOpen && vehicleSearchQuery.length >= 3}
            onOpen={() => {
              if (vehicleSearchQuery.length >= 3) setVehicleDropdownOpen(true);
            }}
            onClose={() => setVehicleDropdownOpen(false)}
    loading={vehicleSearchLoading}
    value={selectedVehicles}
    isOptionEqualToValue={(option, value) => option.id === value.id}
    noOptionsText="Type at least 3 characters to search"
    limitTags={1} // Limit chips shown to avoid height growing
    renderInput={(params) => (
      <TextField
        {...params}
        size="small"
        placeholder=""
        InputProps={{
          ...params.InputProps,
          sx: {
            minHeight: '50px !important', // lock full box height
            padding: '5px 10px !important',
            backgroundColor: 'white !important',
            border: '1px solid pale grey !important',
            borderRadius: '4px !important',
            boxShadow: 'none !important',
            alignItems: 'center !important',
            '& .MuiAutocomplete-tag': {
              maxHeight: '26px',
            },
          }
        }}
        sx={{
          '& .MuiOutlinedInput-notchedOutline': {
            border: 'none !important',
          },
          '& .MuiInputBase-input': {
            padding: '8px 0px !important',
            height: '30px !important',
            border: '1px solid white !important',
            borderRadius: '4px !important',

          },
        }}
      />
    )}
  />
</FormControl>

        {/* Workforce Autocomplete */}
        <FormControl fullWidth sx={{ mb: 2 }}>
          <label>Workforce</label>
          <Autocomplete
            multiple
            options={workforceSearchResults}
            getOptionLabel={(option) => option.name || ''}
            inputValue={workforceSearchQuery}
            onInputChange={(_, newValue, reason) => {
              if (reason === 'input') setWorkforceSearchQuery(newValue);
            }}
            onChange={(_, newValue) => {
              setSelectedWorkforce(newValue);
              setWorkforceSearchQuery('');
            }}
            filterOptions={(x) => x}
            open={workforceDropdownOpen && workforceSearchQuery.length >= 3}
            onOpen={() => {
              if (workforceSearchQuery.length >= 3) setWorkforceDropdownOpen(true);
            }}
            onClose={() => setWorkforceDropdownOpen(false)}
            loading={workforceSearchLoading}
            value={selectedWorkforce}
            isOptionEqualToValue={(option, value) => option.id === value.id}
            noOptionsText="Type at least 3 characters to search"
            renderInput={(params) => (
  <TextField
    {...params}
    size="small"
    placeholder=""
    sx={{
      '& .MuiOutlinedInput-root': {
       backgroundColor: 'white !important',
            border: '1px solid pale grey !important',
        padding: '0 !important',
        height: '48px', // Set desired height
        display: 'flex',
        alignItems: 'center',
      },
      '& .MuiOutlinedInput-notchedOutline': {
        border: 'none !important',
      },
      '& .MuiInputBase-input': {
            padding: '8px 20px !important',
            height: '30px !important',
            border: '1px solid white !important',
            borderRadius: '4px !important',

          },
    }}
  />
)}

          />
        </FormControl>

       <button
  type="submit"
  style={{
    backgroundColor: 'white',
    color: 'green',
    border: '1px solid white',
    padding: '8px 16px',
    borderRadius: '4px',
    cursor: 'pointer',
    fontWeight: 'bold',
    transition: 'all 0.3s ease',
  }}
  onMouseEnter={(e) => {
    e.target.style.backgroundColor = '#4CAF50';
    e.target.style.color = 'white';
  }}
  onMouseLeave={(e) => {
    e.target.style.backgroundColor = 'white';
    e.target.style.color = '#4CAF50';
  }}
>
  {initialData.geofence_id ? 'Update Mapping' : 'Create Mapping'}
</button>
      </Box>
    </div>
  );
};

export default GeofencingRuleMappingForm;