import React, { useState, useEffect } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import GeofencingRuleView from './GeofencingRuleView';
import GeofencingRuleUpdate from './GeofencingRuleUpdate';
import GeofencingRuleDB from './GeofencingRuleDB';

const GEOFRULES_HEADER_TO_KEY_MAP = {
  "Rule Name": { key: "rule_name", label: "Rule Name", sortable: true },
 // "Conditions": { key: "conditions", label: "Conditions", sortable: false },
  "Actions": { key: "action", label: "Actions", sortable: false },
   "Trigger Events": { key: "trigger_events", label: "Trigger Events", sortable: false },

  // "Status": { key: "status", label: "Status", sortable: true },
  "Created Date": { key: "created_date", label: "Created Date", sortable: true },
};

const GeofencingRulesList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const [accountId, setAccountId] = useState(null);
  const [userId, setUserId] = useState(null);
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery
  });
  const [selectedRule, setSelectedRule] = useState(null);
  
  useEffect(() => {
    const storedAccountId = localStorage.getItem('account_id');
    const storedUserId = localStorage.getItem('user_id');
    if (storedAccountId) setAccountId(storedAccountId);
    if (storedUserId) setUserId(storedUserId);
  }, []);

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    { name: 'created_date_from', type: 'date', placeholder: 'From Date' },
    { name: 'created_date_to', type: 'date', placeholder: 'To Date' },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' }
      ]
    }
  ];

  const actionFormats = [
    { label: "view", modal: true, url: "/geofencerules/{id}" },
    { label: "delete", modal: true, url: "/geofencerules/{id}" },
    { label: "edit", modal: true, url: "/geofencerules/{id}" }
  ];

  const handleActionClick = (ruleId, action) => {
    if (["view", "edit", "delete"].includes(action.label)) {
      setSelectedRule({ ruleId, modalType: action.label });
    }
  };

  const buildFetchUrl = () => {
    if (!accountId) return '';
    let url = `/geofencerules/?account_id=${accountId}`;
    if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
    if (searchParams.created_date_from) url += `&from=${searchParams.created_date_from}`;
    if (searchParams.created_date_to) url += `&to=${searchParams.created_date_to}`;
    if (searchParams.status) url += `&status=${searchParams.status}`;
    return url;
  };

  if (!accountId || !userId) return <div>Loading...</div>;

  return (
    <div>
      <SearchPage
        fetchDataUrl={buildFetchUrl()}
        ListComponent={List}
        itemKeyMapping={GEOFRULES_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="id"
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        searchFields={searchFields}
      />

      {selectedRule?.modalType === "view" && (
        <GeofencingRuleView ruleId={selectedRule.ruleId} onClose={() => setSelectedRule(null)} />
      )}
      {selectedRule?.modalType === "edit" && (
        <GeofencingRuleUpdate ruleId={selectedRule.ruleId} onClose={() => setSelectedRule(null)} />
      )}
      {selectedRule?.modalType === "delete" && (
        <GeofencingRuleDB ruleId={selectedRule.ruleId} onClose={() => setSelectedRule(null)} />
      )}
    </div>
  );
};

export default GeofencingRulesList;