// GeoFencingRules/GeofencingRuleView.js
import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';

const TAB_CONFIG = [
  { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
  { id: 'details', label: 'Details', icon: 'info' }
];

const GeofencingRuleView = ({ ruleId, onClose }) => {
  const [ruleData, setRuleData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');
  const [hoveredTab, setHoveredTab] = useState(null);

  useEffect(() => {
    axios.get(`/geofencerules/${ruleId}`)
      .then(res => {
        setRuleData(res.data);
        setLoading(false);
      })
      .catch(err => {
        console.error("Error fetching rule details:", err);
        setError('Error fetching rule details.');
        setLoading(false);
      });
  }, [ruleId]);

  const renderTabContent = () => {
    if (!ruleData) return null;

    switch (activeTab) {
      case 'dashboard':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined">
                dashboard
              </span>
              Rule Summary
            </h4>
            <p><strong>Status:</strong> {ruleData.status || 'Active'}</p>
            <p><strong>Created:</strong> {new Date(ruleData.created_date).toLocaleDateString()}</p>
            <p><strong>Last Updated:</strong> {new Date(ruleData.updated_date).toLocaleDateString()}</p>
          </div>
        );
      case 'details':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined">
                info
              </span>
              Rule Details
            </h4>
            <div style={{ marginTop: '10px' }}>
              <h5>Conditions</h5>
              <ul>
                {ruleData.conditions.map((cond, index) => (
                  <li key={index}>
                    {cond.parameter} {cond.operator} {cond.value}
                  </li>
                ))}
              </ul>
              
              <h5>Actions</h5>
              <ul>
                {ruleData.actions.map((action, index) => (
                  <li key={index}>{action}</li>
                ))}
              </ul>
              
              <h5>Trigger Events</h5>
              <ul>
                {ruleData.trigger_events.map((event, index) => (
                  <li key={index}>
                    {event === '1' ? 'Enter' : event === '0' ? 'Exit' : 'Dwell'}
                  </li>
                ))}
              </ul>
            </div>
          </div>
        );
      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading rule data...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }

  return (
    <Modal onClose={onClose}>
   <div style={{ height: '100%', overflow: 'hidden' }}>
  <div style={{
    display: 'flex',
    alignItems: 'center',
    gap: '20px',
    padding: '0px 10px 10px',
    borderBottom: '1px solid #e0e0e0'
  }}>
    <div style={{
      width: '60px',
      height: '60px',
      borderRadius: '50%',
      backgroundColor: '#e0e0e0',
      color: '#000',
      display: 'flex',
      alignItems: 'center',
      justifyContent: 'center',
      fontSize: '20px',
      fontWeight: 'bold'
    }}>
      {ruleData?.rule_name?.charAt(0)?.toUpperCase() || 'R'}
    </div>
    <div>
      <h3 style={{
        margin: '0px',
        fontWeight: 'bold',
        fontSize: '20px',
        color: 'black'
      }}>
        {ruleData.rule_name}
      </h3>
      <p style={{
        marginTop: '5px',
        fontSize: '16px',
        color: 'black',
        fontWeight: 'bold'
      }}>
        ID : #{ruleData.id}
      </p>
    </div>
  </div>


        <div style={{ display: 'flex', borderBottom: '1px solid #e0e0e0', marginBottom: '20px', padding: '20px +6px' }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '2px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && <span>{tab.label}</span>}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-20px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        <div style={{ padding: '0 20px', overflow: 'hidden' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default GeofencingRuleView;